﻿/* Filesystem functions */

/******************************************************************************/

/* General/shared functions */

function stringValue(array, wide)
{
  var i, step = 1, value, stringArray = [];
  if (wide) step = 2;
  for (i = 0; i + step < array.length + 1; i += step)
  {
    if (wide) value = (array[i] << 8) + array[i + 1];
    else value = array[i];
    if (value == 0) break;
    stringArray.push(String.fromCharCode(value));
  }
  return stringArray.join("");
}

function arrayCompare(array1, array2)
{
  var i;
  if (array1.length != array2.length) return false;
  for (i = 0; i < array1.length; i++)
  {
    if (array1[i] != array2[i]) return false;
  }
  return true;
}

function readNum(input, offset, length, littleEndian, signed)
{
  var i, value = 0;
  for (i = 0; i < length && i < input.length; i++)
  {
    if (littleEndian) value += input[offset + length - i - 1] * Math.pow(2, (length - i - 1) * 8);
    else value += input[offset + i] * Math.pow(2, (length - i - 1) * 8);
  }
  if (signed && value >= Math.pow(2, length * 8 - 1)) value -= Math.pow(2, length * 8);
  return value;
}

/******************************************************************************/

/* GPD (XDBF) functions */
/* http://www.free60.org/XDBF */

function XDBF(data)
{
  var i, titleSpecificData, entryDataOffset, baseOffset, entryOffset, entryLength, entryData, score, flags, entryId, entryIndex, titleSpecificDataLength;
  
  /* XDBF magic bytes ("XDBF") and version */
  if (readNum(data, 0x00, 4) != 0x58444246 || getNum(data, 0x04, 4) != 0x00010000) throw new Error("Not a valid GPD file.");
  
  this.entryCount = readNum(data, 0x0C, 4);
  this.entryOffset = [];
  this.titleSpecificDataLength = [];
  this.achievementScoreAchieved = 0;
  this.achievementTotalScore = 0;
  this.achievementsAchieved = 0;
  this.achievementCount = 0;
  titleSpecificData = [];
  /* Entry Data Offset = Entry Table Length * 0x12 + Free Space Table Length * 0x08 + 0x18 */
  entryDataOffset = readNum(data, 0x08, 4) * 0x12 + readNum(data, 0x10, 4) * 0x08 + 0x18;
  for (i = 0; i < this.entryCount; i++)
  {
    baseOffset = 0x18 + i * 0x12;
    switch (readNum(data, baseOffset, 2))
    {
      /* GPD Namespace == Achievement */
      case 1:
        entryOffset = readNum(data, baseOffset + 0x0A, 4) + entryDataOffset;
        entryLength = readNum(data, baseOffset + 0x0E, 4);
        entryData = data.subarray(entryOffset, entryOffset + entryLength);
        /* Struct size == 0x1C for non-SyncList/non-SyncData achievement entries */
        if (readNum(entryData, 0x00, 4) == 0x1C)
        {
          score = readNum(entryData, 0x0C, 4);
          flags = readNum(entryData, 0x10, 4);
          this.achievementTotalScore += score;
          this.achievementCount++;
          if (flags & 0x20000)
          {
            this.achievementScoreAchieved += score;
            this.achievementsAchieved++;
          }
        }
        break;
      
      /* GPD Namespace == Image */
      case 2:
        entryId = readNum(data, baseOffset + 0x06, 4);
        /* Title Information == 0x8000 */
        if (entryId == 0x8000)
        {
          entryOffset = readNum(data, baseOffset + 0x0A, 4) + entryDataOffset;
          entryLength = readNum(data, baseOffset + 0x0E, 4);
          this.titleImageData = data.subarray(entryOffset, entryOffset + entryLength);
        }
        break;
      
      /* GPD Namespace == Setting */
      case 3:
        entryId = readNum(data, baseOffset + 0x06, 4);
        /* TitleSpecific1 == 0x63E83FFF, TitleSpecific2 == 0x63E83FFE, TitleSpecific3 == 0x63E83FFD */
        if (entryId >= 0x63E83FFD && entryId <= 0x63E83FFF)
        {
          entryOffset = readNum(data, baseOffset + 0x0A, 4) + entryDataOffset;
          entryLength = readNum(data, baseOffset + 0x0E, 4);
          entryData = data.subarray(entryOffset, entryOffset + entryLength);
          entryIndex = 0x63E83FFF - entryId;
          titleSpecificDataLength = readNum(entryData, 0x10, 4);
          if (readNum(entryData, 0, 4) != entryId || titleSpecificDataLength + 0x18 != entryLength)
          {
            throw new Error("GPD settings file TitleSpecific" + (entryIndex + 1) + " is corrupt.");
          }
          titleSpecificData[entryIndex] = entryData.subarray(0x18, titleSpecificDataLength + 0x18);
          this.entryOffset[entryIndex] = entryOffset;
          this.titleSpecificDataLength[entryIndex] = titleSpecificDataLength;
        }
        break;
      
      /* GPD Namespace == String */
      case 5:
        /* Title Information == 0x8000 */
        if (readNum(data, baseOffset + 0x06, 4) == 0x8000)
        {
          entryOffset = readNum(data, baseOffset + 0x0A, 4) + entryDataOffset;
          entryLength = readNum(data, baseOffset + 0x0E, 4);
          entryData = data.subarray(entryOffset, entryOffset + entryLength);
          this.titleName = stringValue(entryData, true);
        }
        break;
      
      default:
    }
  }
  this.data = data;
  if (titleSpecificData[0] && titleSpecificData[1] && titleSpecificData[2])
  {
    this.titleSpecificData = new Uint8Array(titleSpecificData[0].length + titleSpecificData[1].length + titleSpecificData[2].length);
    this.titleSpecificData.set(titleSpecificData[0]);
    this.titleSpecificData.set(titleSpecificData[1], titleSpecificData[0].length);
    this.titleSpecificData.set(titleSpecificData[2], titleSpecificData[0].length + titleSpecificData[1].length);
  }
}

XDBF.prototype.updateData = function(data)
{
  var i, dataOffset = 0, entryOffset, length;
  for (i = 0; i < this.entryOffset.length; i++)
  {
    entryOffset = this.entryOffset[i] + 0x18;
    length = this.titleSpecificDataLength[i];
    this.data.set(data.subarray(dataOffset, dataOffset + length), entryOffset);
    dataOffset += length;
  }
}

/******************************************************************************/

/* STFS functions */
/* Ported from py360: https://github.com/arkem/py360 */

/* py360/stfs.py */

/*
Secure Transacted File System - A container format found on Xbox 360 XTAF partitions
See http://free60.org/STFS
*/

function BlockHashRecord(blocknum, data, table, record)
{
  // Object containing the SHA1 hash of a block as well as its free/used information and next block
  if (data.length != 0x18) throw new Error("BlockHashRecord data of an incorrect length");
  this.record = record;
  this.table = table;
  this.blocknum = blocknum;
  this.hash = data.subarray(0, 0x14);
  this.info = data[0x14];
  this.nextblock = readNum(data, 0x15, 3); // Big endian, unsigned int,
  if (this.info != 0x00 && this.info != 0x40 && this.info != 0x80 && this.info != 0xC0) // 0x00: "Unused", 0x40: "Freed", 0x80: "Old", 0xC0: "Current"
    throw new Error("BlockHashRecord type is unknown");
};

function FileListing(data)
{
  // Object containing the information about a file in the STFS container
  // Data includes size, name, path and firstblock and atime and utime
  
  this.filename = stringValue(data.subarray(0, 0x28));
  if (this.filename == "") throw new Error("FileListing has empty filename");
  this.isdirectory = (0x80 & data[0x28] == 0x80);
  this.numblocks = readNum(data, 0x29, 3, true); // More little endian madness!
  this.firstblock = readNum(data, 0x2F, 3, true); // And again!
  this.pathindex = readNum(data, 0x32, 2, false, true); // Signedness is important here
  this.size = readNum(data, 0x34, 4);
  this.udate = readNum(data, 0x38, 2);
  this.utime = readNum(data, 0x3A, 2);
  this.adate = readNum(data, 0x3C, 2);
  this.atime = readNum(data, 0x3E, 2);
}

function STFS(data)
{
  // Object representing the STFS container. allfiles dict contains a path to filelisting map
  this.magic = stringValue(data.subarray(0, 4));
  if (this.magic != "CON " && this.magic != "PIRS" && this.magic != "LIVE") throw new Error("Not a valid STFS file.");
  
  this.table_spacing = [[0xAB, 0x718F, 0xFE7DA],  // The distance in blocks between tables
                        [0xAC, 0x723A, 0xFD00B]]; // For when tables are 1 block and when they are 2 blocks
  this.block_level = [0xAA, 0x70E4];
  this.data = data;
  this.parse_header(data);
  this.parse_filetable();
}

STFS.prototype.read_filetable = function(firstblock, numblocks)
{
  var i, buf, info, block, blockhash;
  // Given the length and start of the filetable return all its data
  buf = new Uint8Array(numblocks * 0x1000);
  info = 0x80;
  block = firstblock;
  for (i = 0; i < numblocks; i++)
  {
    buf.set(this.read_block(this.fix_blocknum(block), 0x1000), i * 0x1000);
    blockhash = this.get_blockhash(block, 0);
    if (this.table_size_shift > 0 && blockhash.info < 0x80)
      blockhash = this.get_blockhash(block, 1);
    block = blockhash.nextblock;
    info = blockhash.info;
  }
  return buf;
}

STFS.prototype.parse_filetable = function()
{
  var i, data, fl, a, path_components;
  // Generate objects for all the filelistings
  this.filelistings = [];
  this.allfiles = [];
  data = this.read_filetable(this.filetable_blocknumber, this.filetable_blockcount);
  
  for (i = 0; i < data.length; i += 0x40) // File records are 0x40 length
  {
    try
    {
      this.filelistings.push(new FileListing(data.subarray(i, i + 0x40)));
    }
    catch (AssertionError)
    {
      continue;
    }
  }
  for (i = 0; i < this.filelistings.length; i++) // Build a dictionary to access filelistings by path
  {
    fl = this.filelistings[i];
    path_components = [fl.filename];
    while (fl.pathindex != -1 && fl.pathindex < this.filelistings.length)
    {
      try
      {
        fl_path = this.filelistings[fl_path.pathindex];
        path_components.push(fl_path.filename);
      }
      catch (IndexError)
      {
        throw new Error("IndexError: " + this.filename + " " + fl_path.pathindex + " " + this.filelistings.length);
      }
    }
    path_components.push("");
    path_components.reverse();
    this.allfiles[path_components.join("/")] = fl;
  }
}

STFS.prototype.read_file = function(filelisting, size)
{
  var buf, bufPos, block, info, readlen, blockhash;
  // Given a filelisting object return its data
  // This requires checking each blockhash to find the next block.
  // In some cases this requires checking two different hash tables.
  
  if (size == -1)
    size = filelisting.size;
  buf = new Uint8Array(size);
  bufPos = 0;
  block = filelisting.firstblock;
  info = 0x80;
  while (size > 0 && block > 0 && block < this.allocated_count && info >= 0x80)
  {
    readlen = Math.min(0x1000, size);
    buf.set(this.read_block(this.fix_blocknum(block), readlen), bufPos);
    bufPos += 0x1000;
    size -= readlen;
    blockhash = this.get_blockhash(block, 0) // TODO: Optional concurrent verification of blocks
    // If there are multiple tables and the block is free or unused, try other table
    // TODO: There may be times where both tables show allocated blocks yet only one was correct
    //       It would be better to calculate the best chain of blocks, perhaps precalculate like Partition
    if (this.table_size_shift > 0 && blockhash.info < 0x80)
      blockhash = this.get_blockhash(block, 1);
    // if (!this.verify_block(blockhash))
    // {
      // alert("!");
    // }
    block = blockhash.nextblock;
    info = blockhash.info;
  }
  return buf;
}

STFS.prototype.write_file = function(filelisting, size, data)
{
  var bufPos, block, info, readlen, blockhash;
  // Given a filelisting object set its data
  // This requires checking each blockhash to find the next block.
  // In some cases this requires checking two different hash tables.
  
  if (size == -1)
    size = filelisting.size;
  bufPos = 0;
  block = filelisting.firstblock;
  info = 0x80;
  while (size > 0 && block > 0 && block < this.allocated_count && info >= 0x80)
  {
    readlen = Math.min(0x1000, size);
    this.write_block(this.fix_blocknum(block), data.subarray(bufPos, readlen));
    bufPos += 0x1000;
    size -= readlen;
    blockhash = this.get_blockhash(block, 0); // TODO: Optional concurrent verification of blocks
    // If there are multiple tables and the block is free or unused, try other table
    // TODO: There may be times where both tables show allocated blocks yet only one was correct
    //       It would be better to calculate the best chain of blocks, perhaps precalculate like Partition
    if (this.table_size_shift > 0 && blockhash.info < 0x80)
      blockhash = this.get_blockhash(block, 1);
    // if (this.table_size_shift > 0 && blockhash.info < 0x80)
      // blockhash = this.update_blockhash(block, 1, sha1(this.read_block(this.fix_blocknum(block), 0x1000)));
    // else
      // blockhash = this.update_blockhash(block, 0, sha1(this.read_block(this.fix_blocknum(block), 0x1000)));
    block = blockhash.nextblock;
    info = blockhash.info;
  }
}

STFS.prototype.get_hashtable_num = function(blocknum, table_offset)
{
  var record, tablenum;
  // Given a block number return the hash object that goes with it
  record = blocknum % 0xAA;
  // Num tables * space blocks between each (0xAB or 0xAC for [0])
  tablenum = Math.floor(blocknum / 0xAA) * this.table_spacing[this.table_size_shift][0];
  if (blocknum >= 0xAA)
  {
    tablenum += (Math.floor(blocknum / 0x70E4) + 1) << this.table_size_shift // skip level 1 tables 
    if (blocknum >= 0x70E4)
      tablenum += 1 << this.table_size_shift // If we're into level 2 add the level 2 table
  }
  
  // Read the table block, get the correct record and pass it to BlockHashRecord
  
  // Fix to point at the first table (these numbers are offset from data block numbers)
  tablenum += table_offset - (1 << this.table_size_shift);
  return tablenum;
}

STFS.prototype.get_blockhash = function(blocknum, table_offset)
{
  var tablenum, record, hashdata;
  record = blocknum % 0xAA;
  tablenum = this.get_hashtable_num(blocknum, table_offset);
  hashdata = this.read_block(tablenum, 0x1000);
  return new BlockHashRecord(blocknum, hashdata.subarray(record * 0x18, record * 0x18 + 0x18), tablenum, record);
}

STFS.prototype.update_blockhash = function(blocknum, table_offset, hash)
{
  var record, tablenum, hashdata;
  record = blocknum % 0xAA;
  tablenum = this.get_hashtable_num(blocknum, table_offset);
  hashdata = this.read_block(tablenum, 0x1000);
  hashdata.set(hash, record * 0x18);
  this.write_block(tablenum, hashdata);
  return new BlockHashRecord(blocknum, hashdata.subarray(record * 0x18, record * 0x18 + 0x18), tablenum, record);
}

STFS.prototype.rehash_all = function()
{
  var block, buf, blockhash, offset, block_offset, hash;
  for (block = 0; block >= 0 && block < this.allocated_count; block++)
  {
    buf = this.read_block(this.fix_blocknum(block), 0x1000);
    blockhash = this.get_blockhash(block, 0);
    if (blockhash.info >= 0x80)
      this.update_blockhash(block, 0, sha1(buf));
  }
  offset = (this.table_size_shift > 0 ? 0xA000 : 0xB000);
  // Top Hash Table Hash
  block = this.GetTopHashTable(0);
  block_offset = block * 0x1000 + offset;
  buf = this.data.subarray(block_offset, block_offset + 0x1000);
  this.data.set(sha1(buf), 0x037A + 7);
  // Header SHA1 Hash
  this.data.set(sha1(this.data.subarray(0x0344, offset)), 0x032C);
  // Signature
  if (this.magic != "CON ") throw new Error("Resigning LIVE/PIRS files not supported.");
  buf = this.data.subarray(0x022C, 0x022C + 0x0118);
  hash = sha1(buf);
  
  alert("STFS signing is not yet implemented. You will need to open this save in another editor and manually re-sign it.");
}

STFS.prototype.GetTopHashTable = function(block)
{
  var level, num;
  if (block >= 0x4AF768)
    num = 0xFFFFFF;
  else
  {
    if (this.allocated_count <= this.block_level[0])
    {
      // Get Base Level 0 Table
      num = Math.floor(block / this.block_level[0]) * this.table_spacing[this.table_size_shift][0];
      // Adjusts the result for Level 1 table count
      if (block >= this.block_level[0])
      {
        num += (Math.floor(block / this.block_level[1]) + 1) << this.table_size_shift;
        // Adjusts for the Level 2 table
        if (block >= this.block_level[1])
          num += 1 << this.table_size_shift;
      }
      level = 0;
    }
    else if (this.allocated_count <= this.block_level[1])
    {
      // Grab the number of Table 1 blocks
      if (block < this.block_level[1])
        num = this.table_spacing[this.table_size_shift][0];
      else num = this.table_spacing[this.table_size_shift][1] * Math.floor(block / this.block_level[1]) + (1 << this.table_size_shift);
      level = 1;
    }
    else
    {
      num = this.table_spacing[this.table_size_shift][1];
      level = 2;
    }
  }
  return num;
}

STFS.prototype.verify_block = function(blockhash)
{
  // Check the data in the block versus its recorded hash
  return arrayCompare(blockhash.hash, sha1(this.read_block(this.fix_blocknum(blockhash.blocknum), 0x1000)));
}

STFS.prototype.fix_blocknum = function(block_num)
{
  /*
    Given a blocknumber calculate the block on disk that has the data taking into account hash blocks.
    Every 0xAA blocks there is a hash table and depending on header data it
    is 1 or 2 blocks long [((this.header_size+0xFFF) & 0xF000) >> 0xC 0xB == 0, 0xA == 1]
    After 0x70e4 blocks there is another table of the same size every 0x70e4
    blocks and after 0x4af768 blocks there is one last table. This skews blocknumber to offset calcs.
    This is the part of the Free60 STFS page that needed work
  */
  var block_adjust = 0;
  
  if (block_num >= 0xAA)
    block_adjust += (Math.floor(block_num / 0xAA)) + 1 << this.table_size_shift;
  if (block_num > 0x70E4)
    block_adjust += (Math.floor(block_num / 0x70E4) + 1) << this.table_size_shift;
  return block_adjust + block_num;
}

STFS.prototype.read_block = function(blocknum, length)
{
  // Read a block given its block number
  // If reading data blocks call fix_blocknum first
  return this.data.subarray(0xC000 + blocknum * 0x1000, 0xC000 + blocknum * 0x1000 + length);
}

STFS.prototype.write_block = function(blocknum, data)
{
  // Write a block given its block number
  // If writing data blocks call fix_blocknum first
  return this.data.set(data, 0xC000 + blocknum * 0x1000);
}

// This is a huge, messy struct parsing function.
// There is almost no logic here, just offsets.
STFS.prototype.parse_header = function(data)
{
  // Parse the huge STFS header
  if (data.length < 0x971A) throw new Error("STFS Data Too Short");
  //this.magic = stringValue(data.subarray(0, 4));
  if (this.magic == "CON ")
  {
    this.console_id = data.subarray(6, 11);
    this.console_part_number = data.subarray(0xB, 0x14);
    this.console_type = data[0x1F]; // 0x02 is RETAIL, 0x01 is DEVKIT
    this.certificate_date = data.subarray(0x20, 0x28);
    // Not using the certificate at the moment so this blob has:
    // Exponent, modulus, cert signature, signature
    this.certificate_blob = data.subarray(0x28, 0x1AC + 0x80);
  }
  else
    this.certificate_blob = data.subarray(0x4, 0x104);
  
  //this.license_entries = data[0x22C:0x22C:0x100]
  this.content_id = data.subarray(0x32C, 0x32C + 0x14); // Header SHA1 Hash
  this.header_size = readNum(data, 0x340, 4);
  this.content_type = readNum(data, 0x344, 4);
  this.metadata_version = readNum(data, 0x348, 4);
  this.content_size = readNum(data, 0x34C, 8);
  this.media_id = readNum(data, 0x354, 4);
  this.version = readNum(data, 0x358, 4);
  this.base_version = readNum(data, 0x35C, 4);
  this.title_id = readNum(data, 0x360, 4);
  this.platform = data[0x364];
  this.executable_type = data[0x365];
  this.disc_number = data[0x366];
  this.disc_in_set = data[0x367];
  this.save_game_id = readNum(data, 0x368, 4);
  if (this.magic != "CON ")
    this.console_id = data.subarray(0x36C, 0x36C + 5);
  this.profile_id = data.subarray(0x371, 0x379);
  
  this.volume_descriptor_size = data[0x379];
  this.block_seperation = data[0x37B];
  this.filetable_blockcount = readNum(data, 0x37A + 2, 2, true); // Little Endian. Why?
  this.filetable_blocknumber = readNum(data, 0x37A + 4, 3, true); // Why?!?
  this.tophashtable_hash = data.subarray(0x37A + 7, 0x37A + 7 + 0x14);
  this.allocated_count = readNum(data, 0x37A + 0x1B, 4);
  this.unallocated_count = readNum(data, 0x37A + 0x1F, 4);
  
  this.datafile_count = readNum(data, 0x39D, 4);
  this.datafile_size = readNum(data, 0x3A1, 8);
  this.device_id = data.subarray(0x3FD, 0x3FD + 0x14);
  
  this.display_name = data.subarray(0x411, 0x411 + 0x80); // First locale
  this.display_name_blob = data.subarray(0x411, 0x411 + 0x900); // All locales
  this.display_description = data.subarray(0xD11, 0xD11 + 0x80); // This offset might be wrong, 1 desc got truncated 
  this.display_description_blob = data.subarray(0xD11, 0xD11 + 0x900);
  this.publisher_name = data.subarray(0x1611, 0x1611 + 0x80);
  this.title_name = data.subarray(0x1691, 0x1691 + 0x80);
  
  this.transfer_flags = data[0x1711];
  this.thumbnail_size = readNum(data, 0x1712, 4);
  this.titleimage_size = readNum(data, 0x1716, 4);
  this.thumbnail = data.subarray(0x171A, 0x171A + this.thumbnail_size);
  this.titleimage = data.subarray(0x571A, 0x571A + this.titleimage_size);
  
  if (this.metadata_version == 2)
  {
    this.series_id = data.subarray(0x3B1, 0x3B1 + 0x10);
    this.season_id = data.subarray(0x3C1, 0x3C1 + 0x10);
    this.season_number = readNum(data, 0x3D1, 2);
    this.episode_number = readNum(data, 0x3D3, 2);
    this.additional_display_names = data.subarray(0x541A, 0x541A + 0x300);
    this.additional_display_descriptions = data.subarray(0x941A, 0x941A + 0x300);
  }
  
  // Are the hash tables 1 or 2 blocks long?
  if (((this.header_size + 0xFFF) & 0xF000) >> 0xC == 0xB)
    this.table_size_shift = 0;
  else
    this.table_size_shift = 1;
}
