﻿/*

  XML Xbox 360 Save Editor
  Copyright (C) 2012, 2013 absurdlyobfuscated
  
  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program. If not, see <http://www.gnu.org/licenses/>.
  
  No rights are reserved for the save file format of respective games,
  including offsets and other numeric data, text, layout, and icons.

*/

/*
  TODO:
    Merge bitmaskDataItem with bitmaskData
    STFS RSA/signing
    DS: default in inventory index value
*/

var gameSelections = [
["Magic 2014 (Profile)", "magic2014-gpd.xml", "m14g"],
["Magic 2014 (Deck Save)", "magic2014-stfs.xml", "m14s"],
["Halo 4", "halo4.xml", "h4"],
["Magic 2013", "magic2013.xml", "m13"],
["Soul Calibur V", "scv.xml", "scv"],
["Dark Souls", "ds.xml", "ds"],
["Magic 2012", "magic2012.xml", "m12"],
["DOOM", "doom.xml", "doom"],
["DOOM 2", "doom2.xml", "doom2"],
//["Civilization Revolution", "civrev.xml", "civrev"],
["GTA 4", "gta4.xml", "gta4"],
["Gears of War", "gow1.xml", "gow1"],
["Perfect Dark Zero", "pdz.xml", "pdz"],
];

var saveFileFormat;
var saveFileData = [];
var metaItemIds;
var listBlocks = [];
var loadingFinished;

function getNum(input, offset, length, littleEndian)
{
  var i, value = 0;
  offset = parseInt(offset);
  length = parseInt(length);
  for (i = 0; i < length && i < input.length; i++)
  {
    if (littleEndian) value += input[offset + length - i - 1] * Math.pow(2, (length - i - 1) * 8);
    else value += input[offset + i] * Math.pow(2, (length - i - 1) * 8);
  }
  return value;
}

function getHexStr(input, offset, length, littleEndian, noPrefix)
{
  var i, hexStr = "", valuePair;
  offset = parseInt(offset);
  length = parseInt(length);
  for (i = 0; i < length && i < input.length; i++)
  {
    if (littleEndian) valuePair = input[offset + length - i - 1].toString(16);
    else valuePair = input[offset + i].toString(16);
    if (valuePair.length == 1) valuePair = "0" + valuePair;
    hexStr += valuePair;
  }
  if (noPrefix) return hexStr.toUpperCase();
  return "0x" + hexStr.toUpperCase();
}

function toHexStr(str, isHex, noPrefix)
{
  if (str.substr(0, 2).toLowerCase() != "0x")
  {
    if (!isHex) str = parseInt(str).toString(16).toUpperCase();
    if (!noPrefix) str = "0x" + str;
  }
  else if (noPrefix) str = str.substr(2);
  return str;
}

function getArray(input)
{
  if (Array.isArray(input)) return input;
  if (!defined(input)) return [];
  return new Array(input);
}

function getIndex(specifiedIndex, counterIndex)
{
  return (defined(specifiedIndex) ? specifiedIndex : counterIndex);
}

function defined(variable)
{
  return (typeof variable != "undefined");
}

function insertElementNodes(parentElement, elements, before)
{
  var index;
  if (parentElement == null || elements == null) return;
  if (Array.isArray(elements))
  {
    for (index in elements)
    {
      insertElementNodes(parentElement, elements[index]);
    }
  }
  else
  {
    if (before) parentElement.insertBefore(elements, before);
    else parentElement.appendChild(elements);
  }
}

function appendIndexedElements(element, elementArray, index)
{
  if (!defined(index)) index = 0;
  if (!defined(elementArray[index])) elementArray[index] = [element];
  else
  {
    if (Array.isArray(elementArray[index])) elementArray[index].push(element);
    else elementArray[index] = [elementArray[index], element];
  }
}

function mergeAttributes(attributes, newAttributes)
{
  var attribute;
  for (attribute in newAttributes)
  {
    attributes[attribute] = newAttributes[attribute];
  }
  return attributes;
}

function appendChild(element, child)
{
  var index;
  if (!element || child == null) return;
  if (Array.isArray(child))
  {
    for (index in child)
    {
      appendChild(element, child[index]);
    }
  }
  else
  {
    switch (typeof child)
    {
      case "string":
        element.appendChild(document.createTextNode(child));
        break;
      
      case "object":
        element.appendChild(child);
        break;
      
      default:
    }
  }
}

function clearChildren(element)
{
  if (!element) return;
  while (element.firstChild)
  {
    element.removeChild(element.firstChild);
  }
}

function addElement(tagName, attributes, childElements)
{
  var i, element, attribute;
  element = document.createElement(tagName);
  for (attribute in attributes)
  {
    element.setAttribute(attribute, attributes[attribute]);
  }
  if (childElements) appendChild(element, childElements);
  return element;
}

function filterParent(parentIds, ids)
{
  var i, j, newIds = [];
  if (parentIds && parentIds.length > 0)
  {
    for (i = 0; i < ids.length; i++)
    {
      for (j = 0; j < parentIds.length; j++)
      {
        if (parentIds[j].id == ids[i].id)
        {
          newIds.push(ids[i]);
          break;
        }
      }
    }
    return newIds;
  }
  else return ids;
}

function addMetaId(metaItemId, elementId)
{
  if (metaItemIds[metaItemId]) metaItemIds[metaItemId].push(elementId);
  else metaItemIds[metaItemId] = [elementId];
}

function suggestedValue(id, value, format)
{
  var element;
  switch (format)
  {
    case "time-seconds":
      element = document.getElementById(id + "h");
      if (element) element.value = Math.floor(value / 3600);
      element = document.getElementById(id + "m");
      if (element) element.value = Math.floor(value / 60) % 60;
      element = document.getElementById(id + "s");
      if (element) element.value = value % 60;
      break;
    
    case "time-sixtieth-seconds":
      element = document.getElementById(id + "m");
      if (element) element.value = Math.floor(value / 3600);
      element = document.getElementById(id + "s");
      if (element) element.value = Math.floor(value / 60) % 60;
      element = document.getElementById(id + "ss");
      if (element) element.value = value % 60;
      break;
    
    case "time-centiseconds":
      element = document.getElementById(id + "m");
      if (element) element.value = Math.floor(value / 6000);
      element = document.getElementById(id + "s");
      if (element) element.value = Math.floor(value / 100) % 60;
      element = document.getElementById(id + "cs");
      if (element) element.value = value % 100;
      break;
    
    case "time-milliseconds":
      element = document.getElementById(id + "h");
      if (element) element.value = Math.floor(value / 3600000);
      element = document.getElementById(id + "m");
      if (element) element.value = Math.floor(value / 60000) % 60;
      element = document.getElementById(id + "s");
      if (element) element.value = Math.floor(value / 1000) % 60;
      element = document.getElementById(id + "ms");
      if (element) element.value = value % 1000;
      break;
    
    default:
      element = document.getElementById(id);
      if (element) element.value = value;
  }
}

function updateMetaItem(metaIds)
{
  var i, j, total, element, value;
  for (i = 0; i < metaIds.length; i++)
  {
    total = 0;
    for (j = 0; j < metaItemIds[metaIds[i]].length; j++)
    {
      element = document.getElementById(metaItemIds[metaIds[i]][j]);
      if (element)
      {
        value = parseInt(element.value);
        if (!isNaN(value)) total += value;
      }
    }
    element = document.getElementById("meta-" + metaIds[i]);
    if (element) element.value = total;
  }
}

function setAllOption(metaId, index)
{
  var i, element;
  for (i = 0; i < metaItemIds[metaId].length; i++)
  {
    element = document.getElementById(metaItemIds[metaId][i] + index);
    if (element) element.checked = true;
  }
}

function setAllBitmask(metaId, check)
{
  var i, element;
  for (i = 0; i < metaItemIds[metaId].length; i++)
  {
    element = document.getElementById(metaItemIds[metaId][i]);
    if (element) element.checked = check;
  }
}

function setAllMeta(metaId, index)
{
  var i, element;
  for (i = 0; i < metaItemIds[metaId].length; i++)
  {
    element = document.getElementById("meta-" + metaId + "-" + i + "-" + index);
    if (element && element.type == "button") element.click();
  }
}

function setAllSuggested(metaId)
{
  var i, element;
  for (i = 0; i < metaItemIds[metaId].length; i++)
  {
    element = document.getElementById(metaItemIds[metaId][i]);
    if (element && element.type == "button") element.click();
  }
}

function setListItemData(listBlockId, listItemId, listItemPosition, metaItemId, metaSelectionIndex)
{
  var i, j, listBlock, listItemIndex, listItems, metaItems, metaSelections, baseOffset, offset, length, dataHex, element, listElement;
  listBlock = listBlocks[listBlockId];
  listItemIndex = listBlock.listItemIndex[listItemId];
  listItems = getArray(listBlock.listItem);
  metaItems = filterParent(getArray(listItems[listItemIndex].meta), getArray(listBlock.metaItem));
  for (i = 0; i < metaItems.length; i++)
  {
    if (metaItems[i].id == metaItemId)
    {
      metaSelections = getArray(metaItems[i].metaSelection);
      for (j = 0; j < metaSelections.length; j++)
      {
        if (metaSelections[j].index == metaSelectionIndex)
        {
          offset = parseInt(metaSelections[j].offset);
          length = metaSelections[j].length;
          dataHex = metaSelections[j].data;
          break;
        }
      }
      break;
    }
  }
  baseOffset = listBlock.baseOffset + listItemPosition * listBlock.listItemLength;
  insertArray(listBlock.source, baseOffset + offset, length, hexToArray(dataHex, length));
  saveListItem(listBlock, listItemId, listItemPosition, baseOffset);
  listElement = loadListItem(listBlock, listItemId, baseOffset, listItemPosition);
  element = document.getElementById("list-item-" + listBlockId +  "-" + listItemPosition);
  if (element)
  {
    if (element.nextElementSibling) insertElementNodes(element.parentNode, listElement, element.nextElementSibling);
    else insertElementNodes(element.parentNode, listElement);
    element.parentNode.removeChild(element);
  }
}

function addAllListItems(listBlockId)
{
  var i, element, option;
  element = document.getElementById("list-select-" + listBlockId);
  if (!element) return;
  for (i = 0; i < element.options.length; i++)
  {
    option = element.options[i];
    if (!option.disabled && !option.hidden) option.selected = true;
  }
  addListItem(listBlockId);
}

function addListItem(listBlockId)
{
  var i, listBlock, selectElement, option, listItemId, index, indexFound, listItemPosition, newEmptySlots, listItemIndex, listItems, offset, relativeOffset, dataHex, listElement, presentElement;
  listBlock = listBlocks[listBlockId];
  selectElement = document.getElementById("list-select-" + listBlockId);
  if (!selectElement) return;
  for (i = 0; i < selectElement.options.length; i++)
  {
    option = selectElement.options[i];
    if (!option.selected || option.disabled || option.hidden) continue;
    listItemId = option.value;
    if (listBlock.itemCount >= listBlock.maxItems)
    {
      alert("Maximum number of items reached (" + parseInt(listBlock.maxItems) + ").");
      break;
    }
    if (listBlock.type == "sparse")
    {
      indexFound = false;
      newEmptySlots = [];
      for (index in listBlock.emptySlots)
      {
        if (!indexFound)
        {
          listItemPosition = index;
          indexFound = true;
        }
        else newEmptySlots[index] = true;
      }
      listBlock.emptySlots = newEmptySlots;
    }
    else
    {
      listItemPosition = listBlock.itemCount;
    }
    listItemIndex = listBlock.listItemIndex[listItemId];
    listItems = getArray(listBlock.listItem);
    offset = listBlock.baseOffset + listItemPosition * listBlock.listItemLength;
    dataHex = listItems[listItemIndex].defaultData;
    if (!defined(dataHex)) dataHex = "";
    /* Fill in the previously empty item with default/null data */
    insertArray(listBlock.source, offset, listBlock.listItemLength, hexToArray(dataHex, listBlock.listItemLength))
    /* Insert listItemId for this item */
    insertArray(listBlock.source, offset + parseInt(listBlock.itemIdDataItem.offset), listBlock.itemIdDataItem.length, hexToArray(listItemId, listBlock.itemIdDataItem.length))
    listBlock.itemCount++;
    /* Update the list count data item */
    if (listBlock.listCountDataItem)
    {
      relativeOffset = listBlock.baseOffset - parseInt(listBlock.offset);
      insertNum(listBlock.listCountDataItem.source, relativeOffset + parseInt(listBlock.listCountDataItem.offset), listBlock.listCountDataItem.length, listBlock.itemCount);
    }
    if (!listItems[listItemIndex].allowMultiple)
    {
      option.disabled = true;
      option.selected = false;
    }
    listBlock.itemListIndex[listItemPosition] = listItemId;
    loadingFinished = false;
    listElement = loadListItem(listBlock, listItemId, offset, listItemPosition);
    loadingFinished = true;
    presentElement = document.getElementById("list-present-" + listBlockId);
    if (presentElement) insertElementNodes(presentElement, listElement);
  }
  if (presentElement) presentElement.lastElementChild.scrollIntoView(false);
}

function removeAllListItems(listBlockId)
{
  var i, listBlock, element;
  listBlock = listBlocks[listBlockId];
  for (i = listBlock.itemListIndex.length - 1; i >= 0; i--)
  {
    if (listBlock.itemListIndex[i]) removeListItem(listBlockId, listBlock.itemListIndex[i], i);
  }
  element = document.getElementById("list-block-" + listBlockId);
  if (element) element.scrollIntoView(true);
}

function removeListItem(listBlockId, listItemId, listItemPosition)
{
  var i, listBlock, listItems, listItemIndex, baseOffset, offset, element, listElements = [], metaItems, metaSubItems;
  listBlock = listBlocks[listBlockId];
  listItems = getArray(listBlock.listItem);
  listItemPosition = parseInt(listItemPosition);
  baseOffset = listBlock.baseOffset;
  element = document.getElementById("list-present-" + listBlockId);
  if (!element) return;
  element.removeChild(document.getElementById("list-item-" + listBlockId + "-" + listItemPosition));
  if (listBlock.type == "sparse")
  {
    if (defined(listBlock.defaultData))
    {
      /* Fill in item with defaultData */
      insertArray(listBlock.source, baseOffset + listItemPosition * listBlock.listItemLength + parseInt(listBlock.itemIdDataItem.offset), listBlock.listItemLength, hexToArray(listBlock.defaultData, listBlock.listItemLength));
    }
    /* Replace listItemId with emptyItemId for this item */
    insertArray(listBlock.source, baseOffset + listItemPosition * listBlock.listItemLength + parseInt(listBlock.itemIdDataItem.offset), listBlock.itemIdDataItem.length, hexToArray(toHexStr(listBlock.emptyItemId), listBlock.itemIdDataItem.length));
    listBlock.itemCount--;
    listBlock.emptySlots[listItemPosition] = true;
    listBlock.itemListIndex[listItemPosition] = null;
  }
  else
  {
    /* Save list items that will be shifted, and remove them */
    for (i = listItemPosition + 1; i < listBlock.itemCount; i++)
    {
      saveListItem(listBlock, listItemId, i, baseOffset + i * listBlock.listItemLength);
      element.removeChild(document.getElementById("list-item-" + listBlockId + "-" + i));
    }
    /* Shift list items in our saveFileData one position to overwrite the one being removed */
    insertArray(listBlock.source, baseOffset + listItemPosition * listBlock.listItemLength, (listBlock.itemCount - listItemPosition) * listBlock.listItemLength, saveFileData[listBlock.source].subarray(baseOffset + (listItemPosition + 1) * listBlock.listItemLength, baseOffset + (listBlock.itemCount + 1) * listBlock.listItemLength))
    listBlock.itemCount--;
    /* Fill in the end item with null data */
    insertArray(listBlock.source, baseOffset + listBlock.itemCount * listBlock.listItemLength, listBlock.listItemLength, hexToArray((defined(listBlock.defaultData) ? listBlock.defaultData : ""), listBlock.listItemLength));
    /* Reload shifted/removed items */
    for (i = listItemPosition; i < listBlock.itemCount; i++)
    {
      listBlock.itemListIndex[i] = listBlock.itemListIndex[i + 1];
      offset = baseOffset + i * listBlock.listItemLength;
      listElements.push(loadListItem(listBlock, getHexStr(saveFileData[listBlock.source], offset + parseInt(listBlock.itemIdDataItem.offset), listBlock.itemIdDataItem.length), offset, i));
    }
    insertElementNodes(element, listElements);
  }
  if (listBlock.listCountDataItem)
  {
    /* Update the list count data item */
    insertNum(listBlock.listCountDataItem.source, listBlock.listCountDataItem.offset, listBlock.listCountDataItem.length, listBlock.itemCount);
  }
  listItemIndex = listBlock.listItemIndex[listItemId];
  if (listItems[listItemIndex])
  {
    if (!listItems[listItemIndex].allowMultiple)
    {
      element = document.getElementById("list-select-" + listBlockId);
      if (element) element.options[listItemIndex].disabled = false;
    }
    /* Remove nested meta items from metaItemIds */
    metaItems = filterParent(getArray(listItems[listItemIndex].meta), getArray(listBlock.metaItem));
    for (i = 0; i < metaItems.length; i++)
    {
      metaSubItems = getArray(metaItems[i].meta);
      if (metaSubItems.length == 1) metaItemIds[metaSubItems[0].id].pop();
    }
  }
}

function filterMatches(listItem, filters)
{
  var i, j, meta, filterMatched;
  meta = getArray(listItem.meta);
  if (meta.length > 0)
  {
    for (j = 0; j < filters.length; j++)
    {
      if (filters[j].length == 0)
      {
        filterMatched = true;
        break;
      }
      for (k = 0; k < filters[j].length; k++)
      {
        filterMatched = false;
        for (l = 0; l < meta.length; l++)
        {
          if (meta[l].id == filters[j][k])
          {
            filterMatched = true;
            break;
          }
        }
        if (!filterMatched) break;
      }
      if (!filterMatched) break;
    }
  }
  return filterMatched;
}

function loadFilterOptions(listBlock, filters, filterIds, filterIdIndex, filteringItems)
{
  var i, j, k, blockId, ids, activeGroups = [], activeIds = [], groupActive, subFilters, filterElements = [], subFilterElements = [], onclick, onclickFilters, onclickActiveFilters;
  blockId = listBlock.prefix + listBlock.id;
  ids = getArray(filterIds[filterIdIndex]);
  for (i = 0; i < filters.length; i++)
  {
    filters[i].active = false;
    for (j = 0; j < ids.length; j++)
    {
      if (filters[i].id == ids[j])
      {
        filters[i].active = true;
        activeIds.push(filters[i].id);
        activeGroups.push(filters[i].group);
        break;
      }
    }
  }
  for (i = 0; i < filters.length; i++)
  {
    if (filters[i].active)
    {
      onclick = (filteringItems ? "filterPresentList" : "filterAvailableList") + "(\"" + blockId + "\", "
      onclickFilters = [];
      for (k = 0; k < filterIdIndex && k < filterIds.length; k++)
      {
        onclickFilters.push([filterIds[k]]);
      }
      onclickActiveFilters = [];
      for (k = 0; k < activeIds.length; k++)
      {
        if (activeGroups[k] != filters[i].group) onclickActiveFilters.push(activeIds[k]);
      }
      if (onclickActiveFilters.length > 0) onclickFilters.push(onclickActiveFilters);
      onclick += JSON.stringify(onclickFilters) + ");";
      filterElements.push(addElement("div",
                                     {class: "filterActive"},
                                     [filters[i].label,
                                      addElement("input",
                                                 {class: "filterButton",
                                                  type: "button",
                                                  value: "Clear Filter",
                                                  onclick: onclick})]));
      subFilters = getArray(filters[i].filter);
      if (subFilters.length > 0)
      {
        subFilterElements.push(addElement("br"));
        subFilterElements.push(loadFilterOptions(listBlock, subFilters, filterIds, filterIdIndex + 1, filteringItems));
      }
    }
    else
    {
      groupActive = false;
      for (j = 0; j < activeGroups.length; j++)
      {
        if (filters[i].group == activeGroups[j])
        {
          groupActive = true;
          break;
        }
      }
      if (!groupActive)
      {
        onclick = (filteringItems ? "filterPresentList" : "filterAvailableList") + "(\"" + blockId + "\", ";
        onclickFilters = [];
        for (k = 0; k < filterIdIndex && k < filterIds.length; k++)
        {
          onclickFilters.push([filterIds[k]]);
        }
        /* .slice() to copy array by value */
        onclickActiveFilters = activeIds.slice();
        onclickActiveFilters.push(filters[i].id);
        onclickFilters.push(onclickActiveFilters);
        onclick += JSON.stringify(onclickFilters) + ");";
        filterElements.push(addElement("input",
                                       {class: "filterButton",
                                        type: "button",
                                        value: filters[i].label,
                                        onclick: onclick}));
      }
    }
  }
  return filterElements.concat(subFilterElements);
}

function filterPresentList(listBlockId, filterIds)
{
  var i, j, listBlock, element;
  listBlock = listBlocks[listBlockId];
  listBlock.filters = filterIds;
  element = document.getElementById("filter-present-" + listBlockId);
  if (element)
  {
    clearChildren(element);
    insertElementNodes(element, loadFilterOptions(listBlock, getArray(listBlock.filter), filterIds, 0, true));
  }
  element = document.getElementById("list-present-" + listBlockId);
  if (element)
  {
    clearChildren(element);
    insertElementNodes(element, loadListItemsPresent(listBlock));
  }
}

function filterAvailableList(listBlockId, filterIds)
{
  var i, j, listBlock, element;
  listBlock = listBlocks[listBlockId];
  element = document.getElementById("filter-available-" + listBlockId);
  if (element)
  {
    clearChildren(element);
    insertElementNodes(element, loadFilterOptions(listBlock, getArray(listBlock.filter), filterIds, 0, false));
  }
  element = document.getElementById("list-available-" + listBlockId);
  if (element)
  {
    clearChildren(element);
    insertElementNodes(element, loadListItemsAvailable(listBlock, filterIds));
  }
}

function toggleCollapse(element)
{
  var style = element.parentNode.parentNode.nextElementSibling.style;
  if (style.display == "none")
  {
    style.display = "";
    element.textContent  = "[-]";
    element.title  = "Collapse section";
  }
  else
  {
    style.display = "none";
    element.textContent  = "[+]";
    element.title  = "Expand section";
  }
}

function adjustValue(data, dataValue)
{
  var i, valueAdjustment;
  valueAdjustment = getArray(data.valueAdjustment);
  for (i = 0; i < valueAdjustment.length; i++)
  {
    switch (valueAdjustment[i].operator)
    {
      case "*":
        dataValue *= valueAdjustment[i].value;
        break;
      
      case "/":
        dataValue = Math.round(dataValue / valueAdjustment[i].value);
        break;
      
      case "+":
        dataValue += valueAdjustment[i].value;
        break;
      
      case "-":
        dataValue -= valueAdjustment[i].value;
        break;
      
      default:
    }
  }
  return dataValue;
}

function conditionsMet(conditions, source, baseOffset, metaItemIdArray)
{
  var i, j, operand, value = [0, 0], result;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  for (i = 0; i < conditions.length; i++)
  {
    result = false;
    if (!defined(conditions[i].operator)) continue;
    if (conditions[i].leftOperand && conditions[i].rightOperand)
    {
      for (j = 0; j < 2; j++)
      {
        operand = (j == 0 ? conditions[i].leftOperand : conditions[i].rightOperand);
        if (operand.type)
        {
          switch (operand.type.toLowerCase())
          {
            case "dataitem":
              if (!defined(operand.source)) operand.source = source;
              value[j] = getNum(saveFileData[operand.source], baseOffset + parseInt(operand.offset), operand.length);
              break;
            
            case "value":
              if (!operand.value) break;
              switch (operand.value.toLowerCase())
              {
                case "true":
                  value[j] = 1;
                  break;
                
                case "false":
                  value[j] = 0;
                  break;
                
                default:
                  value[j] = parseInt(operand.value);
              }
              break;
            
            case "meta":
              value[j] = metaItemIdArray;
              break;
            
            case "string":
              value[j] = operand.value;
              break;
            
            case "condition":
              value[j] = (conditionsMet(getArray(operand.condition), source, baseOffset, metaItemIdArray) ? 1 : 0);
              break;
            
            default:
          }
        }
      }
    }
    switch (conditions[i].operator.toLowerCase())
    {
      case "equal":
        if (typeof(value[0]) == "object")
        {
          for (j = 0; j < value[0].length; j++)
          {
            if (value[0][j].id == value[1])
            {
              result = true;
              break;
            }
          }
        }
        else if (typeof(value[1]) == "object")
        {
          for (j = 0; j < value[1].length; j++)
          {
            if (value[0] == value[1][j].id)
            {
              result = true;
              break;
            }
          }
        }
        else if (value[0] == value[1]) result = true;
        break;
      
      case "not-equal":
        if (typeof(value[0]) == "object")
        {
          result = true;
          for (j = 0; j < value[0].length; j++)
          {
            if (value[0][j] != value[1])
            {
              result = false;
              break;
            }
          }
        }
        else if (typeof(value[1]) == "object")
        {
          result = true;
          for (j = 0; j < value[1].length; j++)
          {
            if (value[0] != value[1][j])
            {
              result = false;
              break;
            }
          }
        }
        else if (value[0] != value[1]) result = true;
        break;
      
      case "less-than":
        if (value[0] < value[1]) result = true;
        break;
      
      case "greater-than":
        if (value[0] > value[1]) result = true;
        break;
      
      case "less-than-equal":
        if (value[0] <= value[1]) result = true;
        break;
      
      case "greater-than-equal":
        if (value[0] >= value[1]) result = true;
        break;
      
      default:
    }
    if (result && conditions[i].logic == "or") return true;
    if (!result && conditions[i].logic != "or") return false;
  }
  if (conditions.length == 0) result = true;
  return result;
}

function loadNumeric(data, listIdPrefix, index, source, baseOffset, parentMetaItemIds)
{
  var i, j, dataValue, sign, exponentLength, exponent, mantissaLength, mantissa, id, attributes, inputElements, inputAttributes, onchange, onchangeItems, metaItemIdArray, suggestedValues;
  if (!defined(source))
  {
    source = data.source;
    if (!defined(source)) return null;
  }
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  dataValue = getNum(saveFileData[source], baseOffset + parseInt(data.offset), data.length);
  if (data.mask)
  {
    dataValue &= parseInt(data.mask);
  }
  dataValue = adjustValue(data, dataValue);
  if (defined(data.format)) data.format = data.format.toLowerCase();
  switch (data.format)
  {
    case "signed-int":
      if (dataValue >= Math.pow(2, data.length * 8 - 1))
      {
        dataValue -= Math.pow(2, data.length * 8);
      }
      break;
    
    case "float":
      switch (parseInt(data.length))
      {
        case 2: /* 16-bit / half precision */
          exponentLength = 5;
          mantissaLength = 10;
          break;
        
        case 4: /* 32-bit / single precision */
          exponentLength = 8;
          mantissaLength = 23;
          break;
        
        default: /* Unsupported */
          return null;
      }
      sign = (dataValue & Math.pow(2, exponentLength + mantissaLength) ? -1 : 1);
      exponent = (dataValue >> mantissaLength) & (Math.pow(2, exponentLength) - 1);
      mantissa = dataValue & (Math.pow(2, mantissaLength) - 1);
      switch (dataValue)
      {
        case 0:
          break;
        
        case 0xFF:
          if (mantissa) dataValue = NaN;
          else if (sign > 0) dataValue = Number.POSITIVE_INFINITY;
          else dataValue = Number.NEGATIVE_INFINITY;
          break;
        
        default:
          exponent -= Math.pow(2, exponentLength - 1) - 1;
          dataValue = sign * (mantissa + Math.pow(2, mantissaLength)) / Math.pow(2, mantissaLength) * Math.pow(2, exponent);
      }
      break;
    
    case "hex":
      dataValue = toHexStr(dataValue.toString());
      break;
    
    default:
  }
  id = listIdPrefix + "item-" + index;
  inputAttributes = {type: "text"};
  if (data.readOnly) inputAttributes.readonly = "readonly";
  else
  {
    metaItemIdArray = filterParent(parentMetaItemIds, getArray(data.meta));
    if (metaItemIdArray.length > 0)
    {
      onchange = "updateMetaItem(";
      onchangeItems = [];
      for (i = 0; i < metaItemIdArray.length; i++)
      {
        addMetaId(metaItemIdArray[i].id, id);
        onchangeItems.push(metaItemIdArray[i].id);
      }
      onchange += JSON.stringify(onchangeItems) + ");";
      inputAttributes.onchange = onchange;
    }
  }
  switch (data.format)
  {
    case "time-seconds":
      inputElements = [addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMajor",
                                                                    id: id + "h",
                                                                    value: Math.floor(dataValue / 3600),
                                                                    title: "Hours"})),
                       addElement("div", {class: "numericTimeSeparator"}, ":"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "m",
                                                                    value: Math.floor(dataValue / 60) % 60,
                                                                    title: "Minutes"})),
                       addElement("div", {class: "numericTimeSeparator"}, ":"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "s",
                                                                    value: dataValue % 60,
                                                                    title: "Seconds"}))];
      break;
    
    case "time-sixtieth-seconds":
      inputElements = [addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMajor",
                                                                    id: id + "m",
                                                                    value: Math.floor(dataValue / 3600),
                                                                    title: "Minutes"})),
                       addElement("em", null, "'"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "s",
                                                                    value: Math.floor(dataValue / 60) % 60,
                                                                    title: "Seconds"})),
                       addElement("em", null, "\""),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "ss",
                                                                    value: dataValue % 60,
                                                                    title: "1/60th seconds"}))];
      break;
    
    case "time-centiseconds":
      inputElements = [addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMajor",
                                                                    id: id + "m",
                                                                    value: Math.floor(dataValue / 6000),
                                                                    title: "Minutes"})),
                       addElement("em", null, "'"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "s",
                                                                    value: Math.floor(dataValue / 100) % 60,
                                                                    title: "Seconds"})),
                       addElement("em", null, "\""),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinor",
                                                                    id: id + "cs",
                                                                    value: dataValue % 100,
                                                                    title: "Hundreth seconds"}))];
      break;
    
    case "time-milliseconds":
      inputElements = [addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMajorMS",
                                                                    id: id + "h",
                                                                    value: Math.floor(dataValue / 3600000),
                                                                    title: "Hours"})),
                       addElement("div", {class: "numericTimeSeparator"}, ":"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinorMS",
                                                                    id: id + "m",
                                                                    value: Math.floor(dataValue / 60000) % 60,
                                                                    title: "Minutes"})),
                       addElement("div", {class: "numericTimeSeparator"}, ":"),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMinorMS",
                                                                    id: id + "s",
                                                                    value: Math.floor(dataValue / 1000) % 60,
                                                                    title: "Seconds"})),
                       addElement("div", {class: "numericTimeSeparator"}, "."),
                       addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericTimeMicroMS",
                                                                    id: id + "ms",
                                                                    value: dataValue % 1000,
                                                                    title: "Milliseconds"}))];
      break;
    
    default:
      inputElements = [addElement("input",
                                  mergeAttributes(inputAttributes, {class: "numericText",
                                                                    id: id,
                                                                    value: dataValue}))];
  }
  suggestedValues = getArray(data.suggestedValue)
  for (i = 0; i < suggestedValues.length; i++)
  {
    inputAttributes = {class: "suggestedValue",
                       id: id + "-suggested-" + i,
                       type: "button",
                       value: suggestedValues[i].label,
                       onclick: "suggestedValue(\"" + id + "\", \"" + suggestedValues[i].value + "\", \"" + (defined(data.format) ? data.format : "") + "\");"};
    if (suggestedValues[i].info) inputAttributes.title = suggestedValues[i].info;
    inputElements.push(addElement("input", inputAttributes));
    metaItemIdArray = getArray(suggestedValues[i].meta);
    for (j = 0; j < metaItemIdArray.length; j++)
    {
      addMetaId(metaItemIdArray[j].id, id + "-suggested-" + i);
    }
  }
  attributes = {class: "numericItem"};
  if (data.info) attributes.title = data.info;
  return addElement("div",
                    attributes,
                    [addElement("div",
                                {class: "numericLabel"},
                                data.label),
                     addElement("div",
                                {class: "numericValue"},
                                inputElements)]);
}

function loadBitmask(data, listIdPrefix, index, source, baseOffset, parentMetaItemIds)
{
  var i, j, k, dataValue, id, dataItems, attributes, bitmaskElements = [], bitmaskAttributes, metaItemIdArray, bitId, bit;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  dataItems = getArray(data.bitmaskDataItem);
  metaItemIdArray = getArray(data.meta);
  for (i = 0; i < dataItems.length; i++)
  {
    if (!defined(source))
    {
      if (!defined(dataItems[i].source)) continue;
      dataValue = getNum(saveFileData[dataItems[i].source], baseOffset + parseInt(dataItems[i].offset), dataItems[i].length);
    }
    else dataValue = getNum(saveFileData[source], baseOffset + parseInt(dataItems[i].offset), dataItems[i].length);
    dataValue = adjustValue(dataItems[i], dataValue);
    id = listIdPrefix + "item-" + index + "-" + i + "-";
    if (dataItems[i].bits)
    {
      for (j = 0; j < dataItems[i].bits; j++)
      {
        bitId = id + getIndex(dataItems[i].index, j);
        bitmaskAttributes = {class: "bitmaskCheckbox", id: bitId, type: "checkbox"};
        if (dataValue & Math.pow(2, j)) bitmaskAttributes.checked = "checked";
        bitmaskElements.push(addElement("div",
                                        {class: "bitmaskItem"},
                                        [addElement("input",
                                                    bitmaskAttributes),
                                         addElement("label",
                                                    {class: "bitmaskLabel",
                                                     for: bitId},
                                                    data.label + " " + (j + parseInt(dataItems[i].labelOffset)))]));
        for (k = 0; k < metaItemIdArray.length; k++)
        {
          addMetaId(metaItemIdArray[k].id, bitId);
        }
      }
    }
    else
    {
      bit = parseInt(dataItems[i].bit);
      bitId = id + getIndex(dataItems[i].index, bit);
      bitmaskAttributes = {class: "bitmaskCheckbox", id: bitId, type: "checkbox"};
      if (dataValue & Math.pow(2, bit)) bitmaskAttributes.checked = "checked";
      bitmaskElements.push(addElement("div",
                                      {class: "bitmaskItem"},
                                      [addElement("input",
                                                  bitmaskAttributes),
                                       addElement("label",
                                                  {class: "bitmaskLabel",
                                                   for: bitId},
                                                  data.label)]));
      for (k = 0; k < metaItemIdArray.length; k++)
      {
        addMetaId(metaItemIdArray[k].id, bitId);
      }
    }
  }
  attributes = {class: "bitmaskData"};
  if (data.info) attributes.title = data.info;
  return addElement("div",
                    attributes,
                    addElement("div",
                               {class: "bitmaskItems"},
                               bitmaskElements));
}

function loadSelect(data, listIdPrefix, index, source, baseOffset, parentMetaItemIds)
{
  var i, j, dataValue, id, dataItems, optionChoices, metaItemIdArray, selectElements = [], selectAttributes, choiceElements, choiceAttributes, optionElements, optionAttributes, choiceId, choiceFound;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  dataItems = getArray(data.optionDataItem);
  optionChoices = getArray(data.optionChoice);
  metaItemIdArray = getArray(data.meta);
  if (defined(data.format)) data.format = data.format.toLowerCase();
  for (i = 0; i < dataItems.length; i++)
  {
    choiceElements = [];
    if (!defined(source))
    {
      if (!defined(dataItems[i].source)) continue;
      dataValue = getNum(saveFileData[dataItems[i].source], baseOffset + parseInt(dataItems[i].offset), dataItems[i].length);
    }
    else
    {
      dataValue = getNum(saveFileData[source], baseOffset + parseInt(dataItems[i].offset), dataItems[i].length);
    }
    dataValue = adjustValue(dataItems[i], dataValue);
    id = listIdPrefix + "item-" + index + "-" + i + "-";
    /* Associate metaItems from optionData and dataItems, optionData taking precedence */
    if (!data.meta)
    {
      metaItemIdArray = getArray(dataItems[i].meta);
    }
    for (j = 0; j < metaItemIdArray.length; j++)
    {
      addMetaId(metaItemIdArray[j].id, id);
    }
    switch (data.format)
    {
      case "dropdown":
        optionElements = [];
        choiceFound = false;
        for (j = 0; j < optionChoices.length; j++)
        {
          optionAttributes = {value: optionChoices[j].value};
          if (optionChoices[j].info) optionAttributes.title = optionChoices[j].info;
          if (parseInt(optionChoices[j].value) == dataValue)
          {
            optionAttributes.selected = "selected";
            choiceFound = true;
          }
          optionElements.push(addElement("option",
                                         optionAttributes,
                                         optionChoices[j].label));
        }
        /* Only show a blank default entry when none of the other choices are selected */
        if (!choiceFound)
        {
          optionElements.push(addElement("option",
                                         {selected: "selected"}));
        }
        choiceElements.push(addElement("select",
                                       {class: "selectDropdown",
                                        id: id},
                                       optionElements));
        break;
      
      default:
        for (j = 0; j < optionChoices.length; j++)
        {
          choiceId = id + getIndex(optionChoices[j].index, j);
          optionAttributes = {class: "selectRadio",
                              type: "radio",
                              id: choiceId,
                              name: id};
          if (parseInt(optionChoices[j].value) == dataValue) optionAttributes.checked = "checked";
          choiceAttributes = {class: "selectChoiceItem"};
          if (optionChoices[j].info) choiceAttributes.title = optionChoices[j].info;
          choiceElements.push(addElement("div",
                                         choiceAttributes,
                                         [addElement("input",
                                                     optionAttributes),
                                          addElement("label",
                                                     {class: "selectChoiceLabel",
                                                      for: choiceId},
                                                     optionChoices[j].label)]));
        }
    }
    selectAttributes = {class: "selectItem"};
    if (dataItems[i].info) selectAttributes.title = dataItems[i].info;
    selectElements.push(addElement("div",
                                   selectAttributes,
                                   [addElement("div",
                                               {class: "selectLabel"},
                                               dataItems[i].label),
                                    addElement("div",
                                               {class: "selectChoices"},
                                               choiceElements)]));
  }
  return selectElements;
}

function loadText(data, listIdPrefix, index, source, baseOffset, parentMetaItemIds)
{
  var i, dataArray, dataValue = "", offset, length, value, maxLength, id, attributes;
  if (!defined(source))
  {
    source = data.source;
    if (!defined(source)) return null;
  }
  offset = (defined(baseOffset) ? parseInt(baseOffset) : 0) + parseInt(data.offset);
  length = parseInt(data.length);
  dataArray = saveFileData[source].subarray(offset, offset + length);
  if (defined(data.format)) data.format = data.format.toLowerCase();
  switch (data.format)
  {
    case "wide":
      for (i = 0; i < dataArray.length; i += 2)
      {
        value = (dataArray[i] << 8) + dataArray[i + 1];
        if (value == 0) break;
        dataValue += String.fromCharCode(value);
      }
      maxLength = (length - 1) / 2;
      break;
    
    default:
      for (i = 0; i < dataArray.length; i++)
      {
        if (dataArray[i] == 0 || i == dataArray.length - 1)
        {
          dataValue = arrayToString(dataArray.subarray(0, i));
          break;
        }
      }
      maxLength = length - 1;
  }
  id = listIdPrefix + "item-" + index;
  attributes = {class: "textTextBox", type: "text", id: id, maxLength: maxLength, value: dataValue};
  if (data.readOnly) attributes.readonly = "readonly";
  return addElement("div",
                    {class: "textItem"},
                    [addElement("div",
                                {class: "textLabel"},
                                data.label),
                     addElement("div",
                                {class: "textText"},
                                addElement("input",
                                           attributes))]);
}

function loadPicture(data, listIdPrefix, index, source, baseOffset, parentMetaItemIds)
{
  var pictureData, offset, pictureElements = [];
  offset = (defined(baseOffset) ? parseInt(baseOffset) : 0) + parseInt(data.offset);
  if (!defined(source))
  {
    source = data.source;
    if (!defined(source)) return null;
  }
  pictureData = saveFileData[source].subarray(offset, offset + parseInt(data.length));
  if (data.label) pictureElements.push(addElement("div",
                                                  {class: "pictureLabel"},
                                                  data.label));
  pictureElements.push(addElement("img",
                                  {class: "pictureImg",
                                   src: "data:" + data.mimeType + ";base64," + encode64(pictureData)}));
  return addElement("div",
                    {class: "pictureData"},
                    pictureElements);
}

function loadListItemsPresent(listBlock)
{
  var i, isSparse, itemCount, emptyIdHex, itemId, offset, listElements = [];
  isSparse = (listBlock.type == "sparse");
  /* Find itemCount */
  if (isSparse)
  {
    listBlock.itemCount = 0;
    itemCount = listBlock.maxItems;
    emptyIdHex = toHexStr(listBlock.emptyItemId);
  }
  else
  {
    itemCount = getNum(saveFileData[listBlock.listCountDataItem.source], listBlock.baseOffset - parseInt(listBlock.offset) + parseInt(listBlock.listCountDataItem.offset), listBlock.listCountDataItem.length);
    if (listBlock.maxItems && itemCount > listBlock.maxItems) itemCount = listBlock.maxItems;
    listBlock.itemCount = itemCount;
  }
  /* Build presentItems index, build emptySlots index, and load items */
  for (i = 0; i < itemCount; i++)
  {
    offset = listBlock.baseOffset + listBlock.listItemLength * i;
    if (offset > saveFileData[listBlock.source].length) break;
    itemId = getHexStr(saveFileData[listBlock.source], offset + parseInt(listBlock.itemIdDataItem.offset), listBlock.itemIdDataItem.length);
    if (isSparse)
    {
      if (itemId == emptyIdHex)
      {
        listBlock.emptySlots[i] = true;
        continue;
      }
      listBlock.itemCount++;
    }
    listBlock.presentItems[itemId] = true;
    listBlock.itemListIndex[i] = itemId;
    listElements.push(loadListItem(listBlock, itemId, offset, i));
  }
  return listElements;
}

function loadListItem(listBlock, listItemId, offset, listItemPosition)
{
  var blockId, itemIndex, listItems, label, metaItemIdArray, listElements = [], loadedElements = [];
  blockId = listBlock.prefix + listBlock.id;
  if (!defined(listBlock.listItemIndex[listItemId]))
  {
    label = "Unknown item " + listItemId;
    metaItemIdArray = [];
  }
  else
  {
    itemIndex = listBlock.listItemIndex[listItemId];
    listItems = getArray(listBlock.listItem);
    if (listItems[itemIndex].hidden) return null;
    if (listBlock.filters.length > 0 && !filterMatches(listItems[itemIndex], listBlock.filters)) return null;
    label = listItems[itemIndex].label;
    if (listBlock.metaItem) metaItemIdArray = filterParent(getArray(listItems[itemIndex].meta), getArray(listBlock.metaItem));
    else metaItemIdArray = getArray(listItems[itemIndex].meta);
  }
  if (label) listElements.push(addElement("div",
                                          {class: "listItemLabel"},
                                          label));
  loadItems(listBlock, "list-" + blockId + "-" + listItemPosition + "-", 0, loadedElements, listBlock.source, offset, blockId, listItemId, listItemPosition, metaItemIdArray);
  loadedElements.push(addElement("div",
                                 {class: "removeListItem"},
                                 addElement("input",
                                            {class: "removeListItemButton",
                                             type: "button",
                                             value: (listBlock.removeActionText ? listBlock.removeActionText + " " : "Remove ") + (listBlock.itemName ? listBlock.itemName : "Item"),
                                             onclick: "removeListItem(\"" + blockId + "\", \"" + listItemId + "\", \"" + listItemPosition + "\");"})));
  listElements.push(addElement("div",
                               {class: "listItemItems"},
                               loadedElements));
  return addElement("div",
                    {class: "listItem",
                     id: "list-item-" + blockId + "-" + listItemPosition},
                    listElements);
}

function loadListItemsAvailable(listBlock, filters)
{
  var i, blockId, listItems, listElements = [], optionElements = [], optionAttributes, itemId;
  blockId = listBlock.prefix + listBlock.id;
  listItems = getArray(listBlock.listItem);
  for (i = 0; i < listItems.length; i++)
  {
    itemId = toHexStr(listItems[i].id);
    if (filters.length > 0 && !filterMatches(listItems[i], filters)) continue;
    optionAttributes = {value: itemId};
    if (!listItems[i].allowMultiple && listBlock.presentItems[itemId]) optionAttributes.disabled = true;
    if (listItems[i].hidden) optionAttributes.hidden = true;
    optionElements.push(addElement("option",
                                   optionAttributes,
                                   listItems[i].label));
  }
  listElements.push(addElement("select",
                               {class: "listBlockItemSelect",
                                id: "list-select-" + blockId,
                                multiple: "multiple",
                                style: "height: " + ((listItems.length < 20 ? listItems.length * 1.2 : 24) + 0.3) + "em;"},
                               optionElements));
  listElements.push(addElement("input",
                               {class: "listItemAddButton",
                                type: "button",
                                value: (listBlock.addActionText ? listBlock.addActionText : "Add"),
                                onclick: "addListItem(\"" + blockId + "\");"}));
  if (listBlock.allowAddAll) listElements.push(addElement("input",
                                                          {class: "listItemAddButton",
                                                           type: "button",
                                                           value: (listBlock.addActionText ? listBlock.addActionText : "Add") + " All",
                                                           onclick: "addAllListItems(\"" + blockId + "\");"}));
  return listElements;
}

function loadItems(data, listIdPrefix, index, elementIndex, source, baseOffset, listBlockId, listItemId, listItemPosition, metaItemIdArray)
{
  var i, j, assertions, multiBlock, blockItem, listBlock, blockId, listItems, itemId, offset, filters, offsetItem, variableOffsetBlock, conditionalBlock, numericData, bitmaskData, optionData, textData, pictureData, compressedDataBlock, groups, elements, subElements, loadedElements, attributes, metaItems, metaSelections, metaSubItems;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  
  /* Process assertions */
  assertions = getArray(data.assertion);
  for (i = 0; i < assertions.length; i++)
  {
    if (!conditionsMet(getArray(assertions[i].condition), source, baseOffset, metaItemIdArray))
    {
      alert(assertions[i].message);
      throw new Error("assertion");
    }
  }
  
  /* Load multiBlocks */
  multiBlock = getArray(data.multiBlock);
  for (i = 0; i < multiBlock.length; i++)
  {
    elements = []
    blockItem = getArray(multiBlock[i].blockItem);
    for (j = 0; j < blockItem.length; j++)
    {
      subElements = [];
      if (blockItem[j].title) subElements.push(addElement("div",
                                                          {class: "groupTitle"},
                                                          [addElement("div",
                                                                      {class: "groupCollapse"},
                                                                      addElement("a",
                                                                                 {title: "Collapse section",
                                                                                  onclick: "toggleCollapse(this);"},
                                                                                 "[-]")),
                                                           addElement("div",
                                                                      {class: "groupTitleText"},
                                                                      blockItem[j].title)]));
      loadedElements = [];
      index = loadItems(multiBlock[i], listIdPrefix + j + "-", index, loadedElements, blockItem[j].source, baseOffset + parseInt(blockItem[j].offset), "", "", "", getArray(blockItem[j].meta));
      subElements.push(addElement("div",
                                  {class: "groupItem"},
                                  loadedElements));
      elements.push(addElement("div",
                               {class: "group"},
                               subElements));
    }
    appendIndexedElements(elements, elementIndex, multiBlock[i].index);
  }
  
  /* Load listBlocks */
  listBlock = getArray(data.listBlock);
  for (i = 0; i < listBlock.length; i++)
  {
    blockId = listIdPrefix + listBlock[i].id;
    /* Clone list block object in order to maintain unique properties for nested lists */
    listBlocks[blockId] = JSON.parse(JSON.stringify(listBlock[i]));
    listBlocks[blockId].prefix = listIdPrefix;
    listBlocks[blockId].presentItems = [];
    listBlocks[blockId].listItemIndex = [];
    listBlocks[blockId].itemListIndex = [];
    listBlocks[blockId].emptySlots = [];
    listBlocks[blockId].filters = [];
    listBlocks[blockId].baseOffset = baseOffset + parseInt(listBlocks[blockId].offset);
    elements = [];
    subElements = [];
    if (listBlocks[blockId].title) elements.push(addElement("div",
                                                     {class: "listBlockTitle"},
                                                     listBlocks[blockId].title));
    filters = getArray(listBlocks[blockId].filter);
    if (filters.length > 0) subElements.push(addElement("div",
                                                        {class: "filters"},
                                                        [addElement("div",
                                                                    {class: "filterDesc",
                                                                     id: "filter-desc-" + blockId},
                                                                    "Filter:"),
                                                         addElement("div",
                                                                    {class: "filterOptions",
                                                                     id: "filter-present-" + blockId},
                                                                    loadFilterOptions(listBlocks[blockId], filters, [], 0, true))]));
    listItems = getArray(listBlocks[blockId].listItem);
    for (j = 0; j < listItems.length; j++)
    {
      itemId = toHexStr(listItems[j].id);
      listBlocks[blockId].listItemIndex[itemId] = j;
    }
    subElements.push(addElement("div",
                                {class: "listBlockPresentItemList",
                                 id: "list-present-" + blockId},
                                loadListItemsPresent(listBlocks[blockId])));
    if (listBlocks[blockId].allowRemoveAll) subElements.push(addElement("input",
                                                             {class: "removeListItemButton",
                                                              type: "button",
                                                              value: (listBlocks[blockId].removeActionText ? listBlocks[blockId].removeActionText : "Remove") + " All",
                                                              onclick: "removeAllListItems(\"" + blockId + "\");"}));
    elements.push(addElement("div",
                             {class: "listBlockPresentItems"},
                             subElements));
    subElements = [];
    subElements.push(addElement("div",
                                {class: "listItemLabel"},
                                (listBlocks[blockId].addActionText ? listBlocks[blockId].addActionText + " " : "Add ") + (listBlocks[blockId].itemName ? listBlocks[blockId].itemName : "Item")));
    if (filters.length > 0) subElements.push(addElement("div",
                                                        {class: "filters"},
                                                        [addElement("div",
                                                                    {class: "filterDesc",
                                                                     id: "filter-desc-" + blockId},
                                                                    "Filter:"),
                                                         addElement("div",
                                                                    {class: "filterOptions",
                                                                     id: "filter-available-" + blockId},
                                                                    loadFilterOptions(listBlocks[blockId], filters, [], 0, false))]));
    subElements.push(addElement("div",
                                {class: "listBlockAvailableSelect",
                                 id: "list-available-" + blockId},
                                loadListItemsAvailable(listBlocks[blockId], [])));
    elements.push(addElement("div",
                             {class: "listBlockAvailableItems"},
                             addElement("div",
                                        {class: "listItemWide"},
                                        subElements)));
    appendIndexedElements(addElement("div",
                                     {class: "listBlock",
                                      id: "list-block-" + blockId},
                                     elements), elementIndex, listBlocks[blockId].index);
  }
  
  /* Load compressed data blocks */
  compressedDataBlock = getArray(data.compressedDataBlock);
  for (i = 0; i < compressedDataBlock.length; i++)
  {
    offset = baseOffset + parseInt(compressedDataBlock[i].offset);
    if (defined(compressedDataBlock[i].format)) compressedDataBlock[i].format = compressedDataBlock[i].format.toLowerCase();
    switch (compressedDataBlock[i].format)
    {
      case "zlib":
        /* http://tools.ietf.org/html/rfc1950 */
        compressedDataBlock[i].CMF = getNum(saveFileData[compressedDataBlock[i].source], offset, 1);
        compressedDataBlock[i].FLG = getNum(saveFileData[compressedDataBlock[i].source], offset + 1, 1);
        /* Verify FCHECK value */
        if (((compressedDataBlock[i].CMF << 8) + compressedDataBlock[i].FLG) % 31 != 0)
        {
          alert("Compressed data in save does not appear to be valid (header check failed).");
          throw new Error("invalid");
        }
        /* Verify that CM == 8 and CINFO == 7 */
        if (compressedDataBlock[i].CMF != 0x78)
        {
          alert("Compressed data in save does not appear to be valid (unsupported compression method).");
          throw new Error("invalid");
        }
        if (compressedDataBlock[i].uncompressedSizeDataItem)
        {
          compressedDataBlock[i].uncompressedSize = getNum(saveFileData[compressedDataBlock[i].uncompressedSizeDataItem.source], baseOffset + parseInt(compressedDataBlock[i].uncompressedSizeDataItem.offset), compressedDataBlock[i].uncompressedSizeDataItem.length);
        }
        if (compressedDataBlock[i].compressedSizeDataItem)
        {
          compressedDataBlock[i].compressedSize = getNum(saveFileData[compressedDataBlock[i].compressedSizeDataItem.source], baseOffset + parseInt(compressedDataBlock[i].compressedSizeDataItem.offset), compressedDataBlock[i].compressedSizeDataItem.length);
        }
        saveFileData[compressedDataBlock[i].inputIndex] = zip_inflate(saveFileData[compressedDataBlock[i].source].subarray(offset + 2), (compressedDataBlock[i].uncompressedSize ? compressedDataBlock[i].uncompressedSize : (saveFileData[compressedDataBlock[i].source].length - offset) * 50));
        if (compressedDataBlock[i].uncompressedSizeDataItem && saveFileData[compressedDataBlock[i].inputIndex].length != compressedDataBlock[i].uncompressedSize)
        {
          alert("Compressed data in save does not appear to be valid (decompressed to " + saveFileData[compressedDataBlock[i].inputIndex].length + " bytes, should be " + compressedDataBlock[i].uncompressedSize + " bytes).");
          throw new Error("invalid");
        }
        break;
      
      default:
    }
    loadedElements = [];
    index = loadItems(compressedDataBlock[i], listIdPrefix, index, loadedElements, compressedDataBlock[i].inputIndex, 0, listBlockId, listItemId, listItemPosition, getArray(compressedDataBlock[i].meta));
    appendIndexedElements(loadedElements, elementIndex, compressedDataBlock[i].index);
  }
  
  /* Load variable offset blocks */
  variableOffsetBlock = getArray(data.variableOffsetBlock);
  for (i = 0; i < variableOffsetBlock.length; i++)
  {
    offset = baseOffset + (defined(variableOffsetBlock[i].baseOffset) ? parseInt(variableOffsetBlock[i].baseOffset) : 0);
    offsetItem = getArray(variableOffsetBlock[i].offsetItem);
    for (j = 0; j < offsetItem.length; j++)
    {
      offset += getNum(saveFileData[offsetItem[j].source], (offsetItem[j].cumulative ? offset : baseOffset) + parseInt(offsetItem[j].offset), offsetItem[j].length);
    }
    loadedElements = [];
    index = loadItems(variableOffsetBlock[i], listIdPrefix, index, loadedElements, source, offset, listBlockId, listItemId, listItemPosition, getArray(variableOffsetBlock[i].meta));
    appendIndexedElements(loadedElements, elementIndex, variableOffsetBlock[i].index);
  }
  
  /* Load conditional blocks */
  conditionalBlock = getArray(data.conditionalBlock);
  for (i = 0; i < conditionalBlock.length; i++)
  {
    if (conditionsMet(getArray(conditionalBlock[i].condition), source, baseOffset, metaItemIdArray))
    {
      loadedElements = [];
      index = loadItems(conditionalBlock[i], listIdPrefix, index, loadedElements, source, baseOffset, listBlockId, listItemId, listItemPosition, getArray(conditionalBlock[i].meta));
      appendIndexedElements(loadedElements, elementIndex, conditionalBlock[i].index);
    }
  }
  
  /* Load numeric/bitmask/select/text items */
  numericData = getArray(data.numericData);
  for (i = 0; i < numericData.length; i++)
  {
    appendIndexedElements(loadNumeric(numericData[i], listIdPrefix, index++, source, baseOffset, metaItemIdArray), elementIndex, numericData[i].index);
  }
  bitmaskData = getArray(data.bitmaskData);
  for (i = 0; i < bitmaskData.length; i++)
  {
    appendIndexedElements(loadBitmask(bitmaskData[i], listIdPrefix, index++, source, baseOffset, metaItemIdArray), elementIndex, bitmaskData[i].index);
  }
  optionData = getArray(data.optionData);
  for (i = 0; i < optionData.length; i++)
  {
    appendIndexedElements(loadSelect(optionData[i], listIdPrefix, index++, source, baseOffset, metaItemIdArray), elementIndex, optionData[i].index);
  }
  textData = getArray(data.textData);
  for (i = 0; i < textData.length; i++)
  {
    appendIndexedElements(loadText(textData[i], listIdPrefix, index++, source, baseOffset, metaItemIdArray), elementIndex, textData[i].index);
  }
  pictureData = getArray(data.pictureData);
  for (i = 0; i < pictureData.length; i++)
  {
    appendIndexedElements(loadPicture(pictureData[i], listIdPrefix, index++, source, baseOffset, metaItemIdArray), elementIndex, pictureData[i].index);
  }
  
  /* Load groups recursively */
  groups = getArray(data.group);
  for (i = 0; i < groups.length; i++)
  {
    subElements = [];
    loadedElements = [];
    subElements.push(addElement("div",
                                {class: "groupTitle"},
                                [addElement("div",
                                            {class: "groupCollapse"},
                                            addElement("a",
                                                       {title: "Collapse section",
                                                        onclick: "toggleCollapse(this);"},
                                                       "[-]")),
                                 addElement("div",
                                            {class: "groupTitleText"},
                                            groups[i].title)]));
    index = loadItems(groups[i], listIdPrefix, index, loadedElements, source, baseOffset);
    subElements.push(addElement("div",
                                {class: "groupItem"},
                                loadedElements));
    appendIndexedElements(addElement("div",
                                     {class: "group"},
                                     subElements), elementIndex, groups[i].index);
  }
  
  /* Load metaItems */
  metaItems = filterParent(metaItemIdArray, getArray(data.metaItem));
  for (i = 0; i < metaItems.length; i++)
  {
    metaSubItems = getArray(metaItems[i].meta);
    if (defined(metaItems[i].type)) metaItems[i].type = metaItems[i].type.toLowerCase();
    switch (metaItems[i].type)
    {
      case "sum":
        attributes = {class: "metaItemText",
                      type: "text",
                      id: "meta-" + metaItems[i].id,
                      value: "",
                      readonly: "readonly"};
        if (metaItems[i].info) attributes.title = metaItems[i].info;
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         [addElement("div",
                                                     {class: "metaItemLabel"},
                                                     metaItems[i].label),
                                          addElement("input",
                                                     attributes)]), elementIndex, metaItems[i].index);
        break;
      
      case "set-all-option":
        subElements = [];
        if (metaItems[i].label) subElements.push(addElement("div",
                                                            {class: "metaItemLabel"},
                                                            metaItems[i].label));
        metaSelections = getArray(metaItems[i].metaSelection);
        if (!loadingFinished && metaSubItems.length == 1)
        {
          addMetaId(metaSubItems[0].id, listItemPosition);
        }
        for (j = 0; j < metaSelections.length; j++)
        {
          attributes = {class: "metaItemButton",
                        type: "button",
                        value: metaSelections[j].label,
                        onclick: "setAllOption(\"" + metaItems[i].id +  "\", \"" + metaSelections[j].index + "\");"};
          if (metaSubItems.length == 1) attributes.id = "meta-" + metaSubItems[0].id + "-" + (defined(listItemPosition) ? listItemPosition : metaItemIds[metaSubItems[0].id].length - 1) + "-" + getIndex(metaSelections[j].index, j);
          if (metaSelections[j].info) attributes.title = metaSelections[j].info;
          subElements.push(addElement("input",
                                      attributes));
        }
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         subElements), elementIndex, metaItems[i].index);
        break;
      
      case "set-all-bitmask":
        subElements = [];
        if (metaItems[i].label) subElements.push(addElement("div",
                                                             {class: "metaItemLabel"},
                                                             metaItems[i].label));
        metaSelections = getArray(metaItems[i].metaSelection);
        if (!loadingFinished && metaSubItems.length == 1) addMetaId(metaSubItems[0].id, listItemPosition);
        for (j = 0; j < metaSelections.length; j++)
        {
          attributes = {class: "metaItemButton",
                        type: "button",
                        value: metaSelections[j].label,
                        onclick: "setAllBitmask(\"" + metaItems[i].id +  "\", " + (metaSelections[j].check ? "true" : "false") + ");"};
          if (metaSubItems.length == 1) attributes.id = "meta-" + metaSubItems[0].id + "-" + (defined(listItemPosition) ? listItemPosition : metaItemIds[metaSubItems[0].id].length - 1) + "-" + getIndex(metaSelections[j].index, j);
          if (metaSelections[j].info) attributes.title = metaSelections[j].info;
          subElements.push(addElement("input",
                                      attributes));
        }
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         subElements), elementIndex, metaItems[i].index);
        break;
      
      case "set-list-item-data":
        subElements = [];
        subElements.push(addElement("div",
                                    {class: "metaItemLabel"},
                                    metaItems[i].label));
        metaSelections = getArray(metaItems[i].metaSelection);
        if (!loadingFinished && metaSubItems.length == 1) addMetaId(metaSubItems[0].id, listItemPosition);
        for (j = 0; j < metaSelections.length; j++)
        {
          attributes = {class: "metaItemButton",
                        type: "button",
                        value: metaSelections[j].label,
                        onclick: "setListItemData(\"" + listBlockId + "\", \"" + listItemId + "\", \"" + listItemPosition + "\", \"" + metaItems[i].id + "\", \"" + metaSelections[j].index + "\");"};
          if (metaSubItems.length == 1) attributes.id = "meta-" + metaSubItems[0].id + "-" + listItemPosition + "-" + getIndex(metaSelections[j].index, j);
          if (metaSelections[j].info) attributes.title = metaSelections[j].info;
          subElements.push(addElement("input",
                                      attributes));
        }
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         subElements), elementIndex, metaItems[i].index);
        break;
      
      case "set-all-meta":
        subElements = [];
        subElements.push(addElement("div",
                                    {class: "metaItemLabel"},
                                    metaItems[i].label));
        metaSelections = getArray(metaItems[i].metaSelection);
        if (!loadingFinished && metaSubItems.length == 1) addMetaId(metaSubItems[0].id, listItemPosition);
        for (j = 0; j < metaSelections.length; j++)
        {
          attributes = {class: "metaItemButton",
                        type: "button",
                        value: metaSelections[j].label,
                        onclick: "setAllMeta(\"" + metaItems[i].id +  "\", \"" + metaSelections[j].index + "\");"};
          if (metaSubItems.length == 1) attributes.id = "meta-" + metaSubItems[0].id + "-" + (defined(listItemPosition) ? listItemPosition : metaItemIds[metaSubItems[0].id].length - 1) + "-" + getIndex(metaSelections[j].index, j);
          if (metaSelections[j].info) attributes.title = metaSelections[j].info;
          subElements.push(addElement("input",
                                      attributes));
        }
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         subElements), elementIndex, metaItems[i].index);
        break;
      
      case "set-all-suggested":
        subElements = [];
        subElements.push(addElement("div",
                                    {class: "metaItemLabel"},
                                    metaItems[i].label));
        metaSelections = getArray(metaItems[i].metaSelection);
        for (j = 0; j < metaSelections.length; j++)
        {
          attributes = {class: "metaItemButton",
                        type: "button",
                        value: metaSelections[j].label,
                        onclick: "setAllSuggested(\"" + metaSelections[j].id +  "\");"};
          if (metaSelections[j].info) attributes.title = metaSelections[j].info;
          subElements.push(addElement("input",
                                      attributes));
        }
        appendIndexedElements(addElement("div",
                                         {class: "metaItem"},
                                         subElements), elementIndex, metaItems[i].index);
        break;
      
      default:
    }
  }
  
  /* Load other elements */
  elements = getArray(data.br);
  for (i = 0; i < elements.length; i++)
  {
    appendIndexedElements(addElement("br"), elementIndex, elements[i].index);
  }
  
  return index;
}

function load()
{
  var loadedElements = [], element, id;
  metaItemIds = [];
  
  /* Load items recursively, starting with the root saveFileFormat element */
  try
  {
    loadingFinished = false;
    loadItems(saveFileFormat, "", 0, loadedElements);
    loadingFinished = true;
    element = document.getElementById("editor");
    if (element)
    {
      clearChildren(element);
      insertElementNodes(element, loadedElements);
    }
    element = document.getElementById("loadInfoMsg");
    clearChildren(element);
    element = document.getElementById("saveButton");
    if (element) element.style.display = "";
    element = document.getElementById("saveFiles");
    if (element) element.scrollIntoView(true);
  }
  catch (error)
  {
    if (error.message != "assertion") alert("Error loading data.\n\nError: " + error.message + "\nLine: " + error.lineNumber + "\nStack:\n" + error.stack);
    clearChildren(document.getElementById("editor"));
    element = document.getElementById("loadInfoMsg");
    clearChildren(element);
    element = document.getElementById("saveButton");
    if (element) element.style.display = "none";
  }
  
  /* Update metaItems */
  for (id in metaItemIds)
  {
    updateMetaItem([id]);
  }
}

function hexToArray(hexString, length)
{
  var i, hexArray = new Uint8Array(parseInt(length)), hexStringLength;
  hexString = toHexStr(hexString, true, true);
  hexStringLength = Math.floor(hexString.length / 2);
  for (i = 0; i < hexStringLength && i < length; i++)
  {
    hexArray[i] = parseInt(hexString.substr(i * 2, 2), 16);
  }
  for (i = hexStringLength; i < length; i++)
  {
    hexArray[i] = 0;
  }
  return hexArray;
}

function numToArray(value, length)
{
  var i, valueArray = new Uint8Array(parseInt(length));
  if (length > 4)
  {
    for (i = 0; i < length; i++)
    {
      valueArray[length - i - 1] = value & 0xFF;
      value = Math.floor(value / 0x100);
    }
  }
  else
  {
    for (i = 0; i < length; i++)
    {
      valueArray[length - i - 1] = value & 0xFF;
      value >>= 8;
    }
  }
  return valueArray;
}

function insertNum(source, offset, length, value)
{
  var i, valueArray;
  offset = parseInt(offset);
  length = parseInt(length);
  if (value < 0)
  {
    value += Math.pow(2, length * 8);
  }
  valueArray = numToArray(value, length);
  for (i = 0; i < length && i < valueArray.length; i++)
  {
    saveFileData[source][offset + i] = valueArray[i];
  }
}

function insertString(source, offset, length, string)
{
  var i;
  offset = parseInt(offset);
  length = parseInt(length);
  for (i = 0; i < length && i < string.length; i++)
  {
    saveFileData[source][offset + i] = string.charCodeAt(i);
  }
}

function insertArray(source, offset, length, array)
{
  var i;
  saveFileData[source].set(array, offset);
  for (i = array.length; i < length; i++)
  {
    saveFileData[source][i] = 0;
  }
}

function arrayToString(array)
{
  var i, stringArray = [];
  for (i = 0; i < array.length; i++)
  {
    stringArray[i] = String.fromCharCode(array[i]);
  }
  return stringArray.join("");
}

function stringToArray(string, wide)
{
  var i, array;
  if (wide) array = new Uint8Array(string.length * 2 + 2);
  else array = new Uint8Array(string.length + 1);
  for (i = 0; i < string.length; i++)
  {
    if (wide)
    {
      array[i * 2] = string.charCodeAt(i) & 0xFF00;
      array[i * 2 + 1] = string.charCodeAt(i) & 0x00FF;
    }
    else array[i] = string.charCodeAt(i) & 0xFF;
  }
  if (wide)
  {
    array[string.length * 2] = 0;
    array[string.length * 2 + 1] = 0;
  }
  else array[string.length] = 0;
  return array;
}

function makeBlob(data)
{
  var blobBuilder;
  try
  {
    return new Blob([data], {"type" : "application/octet-stream"});
  }
  catch (error)
  {
    blobBuilder = new BlobBuilder();
    blobBuilder.append(data);
    return blobBuilder.getBlob("application/octet-stream");
  }
}

function unadjustValue(data, dataValue)
{
  var i, valueAdjustment;
  valueAdjustment = getArray(data.valueAdjustment);
  for (i = 0; i < valueAdjustment.length; i++)
  {
    switch (valueAdjustment[i].operator)
    {
      case "*":
        dataValue = Math.round(dataValue / valueAdjustment[i].value);
        break;
      
      case "/":
        dataValue *= valueAdjustment[i].value;
        break;
      
      case "+":
        dataValue -= valueAdjustment[i].value;
        break;
      
      case "-":
        dataValue += valueAdjustment[i].value;
        break;
      
      default:
    }
  }
  return dataValue;
}

function validateNumeric(dataValue, element, minimum, maximum)
{
  if (isNaN(dataValue))
  {
    element.focus();
    alert("Value is not a valid number.");
    throw new Error("invalid");
  }
  if (dataValue < minimum || dataValue > maximum)
  {
    element.focus();
    alert("The value " + dataValue + " is out of bounds (should be from " + minimum + " to " + maximum + ", inclusively).");
    throw new Error("invalid");
  }
}

function saveNumeric(data, listIdPrefix, listBlock, listItemId, listItemPosition, index, numericIndex, source, baseOffset)
{
  var id, length, dataValue, dataValueTotal, origValue, signBit = 0, exponentLength, exponent = 0, mantissaLength, mantissa, minimum, maximum, element, dupeItemId;
  if (data.readOnly) return;
  if (!defined(source))
  {
    source = data.source;
    if (!defined(source)) return;
  }
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  id = listIdPrefix + "item-" + index;
  length = data.length;
  if (defined(data.format)) data.format = data.format.toLowerCase();
  switch (data.format)
  {
    case "time-seconds":
      element = document.getElementById(id + "h");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, Math.floor((Math.pow(2, length * 8) - 3600) / 3600));
      dataValueTotal = dataValue * 3600;
      
      element = document.getElementById(id + "m");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue * 60;
      
      element = document.getElementById(id + "s");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue;
      
      dataValue = dataValueTotal;
      break;
    
    case "time-sixtieth-seconds":
      element = document.getElementById(id + "m");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, Math.floor((Math.pow(2, length * 8) - 3600) / 3600));
      dataValueTotal = dataValue * 3600;
      
      element = document.getElementById(id + "s");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue * 60;
      
      element = document.getElementById(id + "ss");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue;
      
      dataValue = dataValueTotal;
      break;
    
    case "time-centiseconds":
      element = document.getElementById(id + "m");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, Math.floor((Math.pow(2, length * 8) - 6000) / 6000));
      dataValueTotal = dataValue * 6000;
      
      element = document.getElementById(id + "s");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue * 100;
      
      element = document.getElementById(id + "cs");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 99);
      dataValueTotal += dataValue;
      
      dataValue = dataValueTotal;
      break;
    
    case "time-milliseconds":
      element = document.getElementById(id + "h");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, Math.floor((Math.pow(2, length * 8) - 3600000) / 3600000));
      dataValueTotal = dataValue * 3600000;
      
      element = document.getElementById(id + "m");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue * 60000;
      
      element = document.getElementById(id + "s");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 59);
      dataValueTotal += dataValue * 1000;
      
      element = document.getElementById(id + "ms");
      if (!element) return;
      dataValue = parseInt(element.value);
      validateNumeric(dataValue, element, 0, 999);
      dataValueTotal += dataValue;
      
      dataValue = dataValueTotal;
      break;
    
    case "float":
      element = document.getElementById(id);
      if (!element) return;
      dataValue = parseFloat(element.value);
      switch (parseInt(data.length))
      {
        case 2: /* 16-bit / half precision */
          exponentLength = 5;
          mantissaLength = 10;
          break;
        
        case 4: /* 32-bit / single precision */
          exponentLength = 8;
          mantissaLength = 23;
          break;
        
        case 8: /* 64-bit / double precision */
          exponentLength = 11;
          mantissaLength = 52;
          break;
        
        default: /* Unsupported */
          return;
      }
      if (dataValue != 0)
      {
        signBit = (dataValue < 0 ? 1 : 0);
        dataValue = Math.abs(dataValue);
        exponent = Math.floor(Math.log(dataValue) / Math.LN2);
        mantissa = dataValue / Math.pow(2, exponent) * Math.pow(2, mantissaLength) - Math.pow(2, mantissaLength);
        dataValue = signBit << exponentLength + mantissaLength | exponent + Math.pow(2, exponentLength - 1) - 1 << mantissaLength | mantissa;
      }
      break;
    
    default:
      element = document.getElementById(id);
      if (!element) return;
      dataValue = parseInt(element.value);
      if (data.format == "signed-int")
      {
        minimum = (data.minimum ? parseInt(data.minimum) : -Math.pow(2, data.length * 8 - 1));
        maximum = (data.maximum ? parseInt(data.maximum) : Math.pow(2, length * 8 - 1) - 1);
      }
      else
      {
        minimum = (data.minimum ? parseInt(data.minimum) : 0);
        maximum = (data.maximum ? parseInt(data.maximum) : Math.pow(2, length * 8) - 1);
      }
      validateNumeric(dataValue, element, minimum, maximum);
  }
  if (listBlock && data.requireUnique)
  {
    if (!listBlock.uniqueValues[numericIndex]) listBlock.uniqueValues[numericIndex] = [];
    dupeItemId = listBlock.uniqueValues[numericIndex][dataValue];
    if (dupeItemId)
    {
      element.focus();
      alert("Value must be unique for all list items.\nThe value " + dataValue + " is also in use by item " + getArray(listBlock.listItem)[listBlock.listItemIndex[dupeItemId]].label + ".");
      throw new Error("invalid");
    }
    listBlock.uniqueValues[numericIndex][dataValue] = listItemId;
  }
  dataValue = unadjustValue(data, dataValue);
  if (data.mask)
  {
    origValue = getNum(saveFileData[source], baseOffset + parseInt(data.offset), length);
    dataValue |= ~parseInt(data.mask) & origValue;
  }
  insertNum(source, baseOffset + parseInt(data.offset), length, dataValue);
  if (data.linkedDataItem)
  {
    insertNum(data.linkedDataItem.source, baseOffset + parseInt(data.linkedDataItem.offset), data.linkedDataItem.length, dataValue);
  }
}

function saveBitmask(data, listIdPrefix, listBlock, listItemId, listItemPosition, index, bitmaskIndex, source, baseOffset)
{
  var i, j, dataItems, dataValue, origValue, inputSource, length, offset, id, bit, element;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  dataItems = getArray(data.bitmaskDataItem);
  for (i = 0; i < dataItems.length; i++)
  {
    length = dataItems[i].length;
    offset = parseInt(dataItems[i].offset);
    if (!defined(source))
    {
      inputSource = dataItems[i].source;
      if (!defined(inputSource)) continue;
    }
    else
    {
      inputSource = source;
    }
    dataValue = getNum(saveFileData[inputSource], baseOffset + offset, length);
    origValue = dataValue;
    id = listIdPrefix + "item-" + index + "-" + i + "-";
    if (dataItems[i].bits)
    {
      for (j = 0; j < dataItems[i].bits; j++)
      {
        element = document.getElementById(id + j);
        if (element)
        {
          if (element.checked)
          {
            dataValue |= Math.pow(2, j);
          }
          else
          {
            dataValue &= ~Math.pow(2, j);
          }
        }
      }
    }
    else
    {
      bit = dataItems[i].bit;
      element = document.getElementById(id + bit);
      if (element)
      {
        if (element.checked)
        {
          dataValue |= Math.pow(2, bit);
        }
        else
        {
          dataValue &= ~Math.pow(2, bit);
        }
      }
    }
    if (dataValue != origValue)
    {
      dataValue = unadjustValue(dataItems[i], dataValue);
      insertNum(inputSource, baseOffset + offset, length, dataValue);
      if (data.linkedDataItem)
      {
        insertNum(data.linkedDataItem.source, baseOffset + parseInt(data.linkedDataItem.offset), data.linkedDataItem.length, dataValue);
      }
    }
  }
}

function saveSelect(data, listIdPrefix, listBlock, listItemId, listItemPosition, index, selectIndex, source, baseOffset)
{
  var i, j, dataItems, optionChoices, inputSource, id, choiceFound, dataValue, element;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  dataItems = getArray(data.optionDataItem);
  optionChoices = getArray(data.optionChoice);
  if (defined(data.format)) data.format = data.format.toLowerCase();
  for (i = 0; i < dataItems.length; i++)
  {
    if (!defined(source))
    {
      inputSource = dataItems[i].source;
      if (!defined(inputSource)) continue;
    }
    else
    {
      inputSource = source;
    }
    id = listIdPrefix + "item-" + index + "-" + i + "-";
    choiceFound = false;
    switch (data.format)
    {
      case "dropdown":
        element = document.getElementById(id);
        if (!element) break;
        dataValue = element.children[element.selectedIndex].value;
        if (dataValue.length > 0)
        {
          choiceFound = true;
        }
        break;
      
      default:
        for (j = 0; j < optionChoices.length; j++)
        {
          element = document.getElementById(id + j);
          if (!element) continue;
          if (element.checked)
          {
            dataValue = optionChoices[j].value;
            choiceFound = true;
            break;
          }
        }
    }
    if (choiceFound)
    {
      dataValue = unadjustValue(dataItems[i], dataValue);
      insertNum(inputSource, baseOffset + parseInt(dataItems[i].offset), dataItems[i].length, dataValue);
      if (data.linkedDataItem)
      {
        insertNum(data.linkedDataItem.source, baseOffset + parseInt(data.linkedDataItem.offset), data.linkedDataItem.length, dataValue);
      }
    }
  }
}

function saveText(data, listIdPrefix, listBlock, listItemId, listItemPosition, index, textIndex, source, baseOffset)
{
  var i, id, length, element, dataValue, value, newDataValue = "";
  if (data.readOnly) return;
  if (!defined(source))
  {
    source = data.source;
    if (!defined(source)) return;
  }
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  id = listIdPrefix + "item-" + index;
  length = parseInt(data.length);
  element = document.getElementById(id);
  if (!element) return;
  dataValue = element.value;
  if (defined(data.format)) data.format = data.format.toLowerCase();
  switch (data.format)
  {
    case "wide":
      for (i = 0; i < dataValue.length; i++)
      {
        value = dataValue.charCodeAt(i);
        newDataValue += String.fromCharCode(value & 0xFF00) + String.fromCharCode(value & 0x00FF);
      }
      dataValue = newDataValue + "\0\0";
      break;
    
    default:
      dataValue += "\0";
  }
  insertString(source, baseOffset + parseInt(data.offset), length, dataValue);
  if (data.linkedDataItem)
  {
    insertString(data.linkedDataItem.source, baseOffset + parseInt(data.linkedDataItem.offset), data.linkedDataItem.length, dataValue);
  }
}

function saveListItem(listBlock, listItemId, listItemPosition, offset)
{
  var listItems, itemIndex, metaItemIdArray, listIdPrefix;
  listItems = getArray(listBlock.listItem);
  itemIndex = listBlock.listItemIndex[listItemId];
  if (!defined(itemIndex)) return;
  if (listBlock.metaItem) metaItemIdArray = filterParent(getArray(listItems[itemIndex].meta), getArray(listBlock.metaItem));
  else metaItemIdArray = getArray(listItems[itemIndex].meta);
  listIdPrefix = "list-" + listBlock.prefix + listBlock.id + "-" + listItemPosition + "-";
  saveItems(listBlock, listIdPrefix, listBlock, listItemId, listItemPosition, 0, listBlock.source, offset, metaItemIdArray);
}

function saveItems(data, listIdPrefix, listBlock, listItemId, listItemPosition, index, source, baseOffset, metaItemIdArray)
{
  var i, j, multiBlock, blockItem, numericData, bitmaskData, optionData, textData, pictureData, listBlockArray, blockId, itemCount, offset, compressedDataBlock, compressionLevel, compressedData, compressedSize, checksum, newSaveFileData, variableOffsetBlock, offsetItem, id, conditionalBlock, groups;
  baseOffset = (defined(baseOffset) ? parseInt(baseOffset) : 0);
  
  /* Save multiBlocks */
  multiBlock = getArray(data.multiBlock);
  for (i = 0; i < multiBlock.length; i++)
  {
    blockItem = getArray(multiBlock[i].blockItem);
    for (j = 0; j < blockItem.length; j++)
    {
      index = saveItems(multiBlock[i], listIdPrefix + j + "-", listBlock, listItemId, listItemPosition, index, blockItem[j].source, baseOffset + parseInt(blockItem[j].offset), getArray(blockItem[j].meta));
    }
  }
  
  /* Save listBlocks */
  listBlockArray = getArray(data.listBlock);
  for (i = 0; i < listBlockArray.length; i++)
  {
    blockId = listIdPrefix + listBlockArray[i].id;
    itemCount = (listBlocks[blockId].type == "sparse" ? listBlocks[blockId].maxItems : listBlocks[blockId].itemCount);
    listBlocks[blockId].uniqueValues = [];
    for (j = 0; j < itemCount; j++)
    {
      offset = baseOffset + parseInt(listBlocks[blockId].offset) + listBlocks[blockId].listItemLength * j;
      id = getHexStr(saveFileData[listBlocks[blockId].source], offset + parseInt(listBlocks[blockId].itemIdDataItem.offset), listBlocks[blockId].itemIdDataItem.length);
      if (id != listBlocks[blockId].emptyItemId)
      {
        saveListItem(listBlocks[blockId], id, j, offset);
      }
    }
  }
  
  /* Save compressed data blocks */
  compressedDataBlock = getArray(data.compressedDataBlock);
  for (i = 0; i < compressedDataBlock.length; i++)
  {
    index = saveItems(compressedDataBlock[i], listIdPrefix, listBlock, listItemId, listItemPosition, index, compressedDataBlock[i].inputIndex, 0, getArray(compressedDataBlock[i].meta));
    offset = baseOffset + parseInt(compressedDataBlock[i].offset);
    if (defined(compressedDataBlock[i].format)) compressedDataBlock[i].format = compressedDataBlock[i].format.toLowerCase();
    switch (compressedDataBlock[i].format)
    {
      case "zlib":
        compressionLevel = 7; //((compressedDataBlock[i].FLG & 0x0C) >> 6) * 3;
        compressedData = zip_deflate(saveFileData[compressedDataBlock[i].inputIndex], compressionLevel);
        compressedSize = (compressedDataBlock[i].compressedSizeDataItem ? compressedDataBlock[i].compressedSize : saveFileData[compressedDataBlock[i].source].length - offset);
        /* compressedData does not include the 2 byte header or 4 byte checksum */
        if (compressedData.length + 2 != compressedSize)
        {
          newSaveFileData = new Uint8Array(saveFileData[compressedDataBlock[i].source].length + compressedData.length + 2 - compressedSize + 4);
          newSaveFileData.set(saveFileData[compressedDataBlock[i].source].subarray(0, offset + 2));
          if (compressedSize + offset > saveFileData[compressedDataBlock[i].source].length)
          {
            newSaveFileData.set(saveFileData[compressedDataBlock[i].source].subarray(compressedSize + offset), compressedSize + offset);
          }
          saveFileData[compressedDataBlock[i].source] = newSaveFileData;
          newSaveFileData = null;
          compressedDataBlock[i].compressedSize = compressedData.length + 6;
        }
        saveFileData[compressedDataBlock[i].source].set(compressedData, offset + 2);
        checksum = adler32(compressedData);
        insertNum(compressedDataBlock[i].source, offset + 2 + compressedData.length, 4, checksum);
        if (compressedDataBlock[i].uncompressedSizeDataItem)
        {
          insertNum(compressedDataBlock[i].uncompressedSizeDataItem.source, baseOffset + parseInt(compressedDataBlock[i].uncompressedSizeDataItem.offset), compressedDataBlock[i].uncompressedSizeDataItem.length, saveFileData[compressedDataBlock[i].inputIndex].length);
        }
        if (compressedDataBlock[i].compressedSizeDataItem)
        {
          insertNum(compressedDataBlock[i].compressedSizeDataItem.source, baseOffset + parseInt(compressedDataBlock[i].compressedSizeDataItem.offset), compressedDataBlock[i].compressedSizeDataItem.length, compressedData.length + 6);
        }
        compressedData = null;
        break;
      
      default:
    }
  }
  
  /* Save variable offset blocks */
  variableOffsetBlock = getArray(data.variableOffsetBlock);
  for (i = 0; i < variableOffsetBlock.length; i++)
  {
    offset = baseOffset + (defined(variableOffsetBlock[i].baseOffset) ? parseInt(variableOffsetBlock[i].baseOffset) : 0);
    offsetItem = getArray(variableOffsetBlock[i].offsetItem);
    for (j = 0; j < offsetItem.length; j++)
    {
      offset += getNum(saveFileData[offsetItem[j].source], (offsetItem[j].cumulative ? offset : baseOffset) + parseInt(offsetItem[j].offset), offsetItem[j].length);
    }
    index = saveItems(variableOffsetBlock[i], listIdPrefix, listBlock, listItemId, listItemPosition, index, source, offset, getArray(variableOffsetBlock[i].meta));
  }
  
  /* Save conditional blocks */
  conditionalBlock = getArray(data.conditionalBlock);
  for (i = 0; i < conditionalBlock.length; i++)
  {
    if (conditionsMet(getArray(conditionalBlock[i].condition), source, baseOffset, metaItemIdArray))
    {
      index = saveItems(conditionalBlock[i], listIdPrefix, listBlock, listItemId, listItemPosition, index, source, baseOffset, metaItemIdArray);
    }
  }
  
  /* Save numeric/bitmask/select items */
  numericData = getArray(data.numericData);
  for (i = 0; i < numericData.length; i++)
  {
    saveNumeric(numericData[i], listIdPrefix, listBlock, listItemId, listItemPosition, index++, i, source, baseOffset);
  }
  bitmaskData = getArray(data.bitmaskData);
  for (i = 0; i < bitmaskData.length; i++)
  {
    saveBitmask(bitmaskData[i], listIdPrefix, listBlock, listItemId, listItemPosition, index++, i, source, baseOffset);
  }
  optionData = getArray(data.optionData);
  for (i = 0; i < optionData.length; i++)
  {
    saveSelect(optionData[i], listIdPrefix, listBlock, listItemId, listItemPosition, index++, i, source, baseOffset);
  }
  textData = getArray(data.textData);
  for (i = 0; i < textData.length; i++)
  {
    saveText(textData[i], listIdPrefix, listBlock, listItemId, listItemPosition, index++, i, source, baseOffset);
  }
  pictureData = getArray(data.pictureData);
  index += pictureData.length;
  
  /* Save groups recursively */
  groups = getArray(data.group);
  for (i = 0; i < groups.length; i++)
  {
    index = saveItems(groups[i], listIdPrefix, listBlock, listItemId, listItemPosition, index, source, baseOffset);
  }
  
  return index;
}

function setSaveEvent()
{
  var i, inputSources;
  document.getElementById("saveButton").value = "Saving...";
  if (saveFileFormat.combinedInputSource) inputSources = getArray(saveFileFormat.combinedInputSource.inputSource);
  else inputSources = getArray(saveFileFormat.inputSource);
  for (i = 0; i < inputSources.length; i++)
  {
    clearChildren(document.getElementById("download" + getIndex(inputSources[i].index, i)));
  }
  setTimeout(save, 20);
}

function save()
{
  var i, j, k, inputSources, checksum, checksums, offset = 0, index, useBlob = false, blob, extraBlob, length, file, output, extraOutput, element, aElement, extraLabel, extraTitle;
  if (saveFileFormat.combinedInputSource) inputSources = getArray(saveFileFormat.combinedInputSource.inputSource);
  else inputSources = getArray(saveFileFormat.inputSource);
  try
  {
    saveItems(saveFileFormat, "", null, "", 0, 0);
    
    for (i = 0; i < inputSources.length; i++)
    {
      index = getIndex(inputSources[i].index, i);
      checksums = getArray(inputSources[i].checksum);
      for (j = 0; j < checksums.length; j++)
      {
        if (defined(checksums[j].type)) checksums[j].type = checksums[j].type.toLowerCase();
        if (checksums[j].defaultValue)
        {
          switch (checksums[j].type)
          {
            case "crc32":
              length = 4;
              break;
            
            case "md5":
              length = 16;
              break;
            
            case "sha1":
              length = 20;
              break;
            
            default:
          }
          insertArray(index, checksums[j].position, length, hexToArray(checksums[j].defaultValue, length))
        }
        
        switch (checksums[j].type)
        {
          case "crc32":
            checksum = crc32(saveFileData[index].subarray(checksums[j].startOffset, parseInt(checksums[j].startOffset) + parseInt(checksums[j].length)));
            insertNum(index, checksums[j].position, 4, checksum);
            break;
          
          case "md5":
            checksum = md5(saveFileData[index].subarray(checksums[j].startOffset, parseInt(checksums[j].startOffset) + parseInt(checksums[j].length)));
            insertArray(index, checksums[j].position, 16, checksum);
            break;
          
          case "sha1":
            checksum = sha1(saveFileData[index].subarray(checksums[j].startOffset, parseInt(checksums[j].startOffset) + parseInt(checksums[j].length)));
            insertArray(index, checksums[j].position, 20, checksum);
            break;
          
          default:
        }
      }
      if (saveFileFormat.combinedInputSource)
      {
        saveFileData[index].set(saveFileData[saveFileFormat.combinedInputSource.index].subarray(offset, offset + parseInt(inputSources[i].combinedSource.length)), inputSources[i].combinedSource.offset);
        offset += parseInt(inputSources[i].combinedSource.length);
      }
      
      window.URL = window.URL || window.webkitURL;
      window.BlobBuilder = window.BlobBuilder || window.WebKitBlobBuilder || window.MozBlobBuilder || window.MSBlobBuilder;
      if (window.URL && (window.BlobBuilder || typeof Blob != "undefined"))
      {
        useBlob = true;
        if (inputSources[i].objectURL) window.URL.revokeObjectURL(inputSources[i].objectURL);
        if (inputSources[i].extraObjectURL) window.URL.revokeObjectURL(inputSources[i].extraObjectURL);
      }
      if (defined(inputSources[i].format)) inputSources[i].format = inputSources[i].format.toLowerCase();
      switch (inputSources[i].format)
      {
        case "xdbf-settings":
          inputSources[i].XDBF.updateData(saveFileData[index]);
          if (useBlob)
          {
            blob = makeBlob(inputSources[i].XDBF.data.buffer);
            extraBlob = makeBlob(saveFileData[index].buffer);
          }
          else
          {
            output = encode64(inputSources[i].XDBF.data);
            extraOutput = encode64(saveFileData[index]);
          }
          extraLabel = "[Raw Data]";
          extraTitle = "Raw TitleSpecific data stored in the GPD file.";
          break;
        
        case "stfs":
          files = getArray(inputSources[i].file);
          file = files[inputSources[i].fileIndex];
          inputSources[i].STFS.write_file(inputSources[i].STFS.allfiles[file.name], -1, saveFileData[index]);
          inputSources[i].STFS.rehash_all();
          if (useBlob)
          {
            blob = makeBlob(inputSources[i].STFS.data.buffer);
            extraBlob = makeBlob(saveFileData[index].buffer);
          }
          else
          {
            output = encode64(inputSources[i].STFS.data);
            extraOutput = encode64(saveFileData[index]);
          }
          extraLabel = "[" + file.name.substr(1) + "]";
          extraTitle = file.name.substr(1) + " file stored in " + inputSources[i].name + ".";
          break;
        
        default:
          if (useBlob) blob = makeBlob(saveFileData[index].buffer);
          else output = encode64(saveFileData[index]);
      }
      if (useBlob)
      {
        inputSources[i].objectURL = window.URL.createObjectURL(blob);
        if (extraBlob) inputSources[i].extraObjectURL = window.URL.createObjectURL(extraBlob);
        
        element = document.getElementById("download" + index);
        aElement = addElement("a",
                              {class: "downloadLink",
                               download: inputSources[i].name,
                               href: inputSources[i].objectURL},
                              "Download " + inputSources[i].name);
        /* 'data-downloadurl' isn't a valid JavaScript identifier and must be set manually */
        aElement.setAttribute("data-downloadurl", "application/octet-stream:" + inputSources[i].name + ":" + inputSources[i].objectURL);
        insertElementNodes(element, aElement);
        if (extraBlob)
        {
          aElement = addElement("a",
                                {class: "downloadLinkExtra",
                                 download: inputSources[i].name + ".rawData",
                                 href: inputSources[i].extraObjectURL,
                                 title: extraTitle},
                                extraLabel);
          aElement.setAttribute("data-downloadurl", "application/octet-stream:" + inputSources[i].name + ".rawData:" + inputSources[i].extraObjectURL);
          insertElementNodes(element, aElement);
        }
      }
      else
      {
        element = document.getElementById("download" + index);
        insertElementNodes(element, addElement("a",
                                               {class: "downloadLink",
                                                download: inputSources[i].name,
                                                href: "data:application/octet-stream;base64," + output},
                                               "Download " + inputSources[i].name));
        if (extraOutput)
        {
          insertElementNodes(element, addElement("a",
                                                 {class: "downloadLinkExtra",
                                                  download: inputSources[i].name + ".rawData",
                                                  href: "data:application/octet-stream;base64," + extraOutput,
                                                  title: extraTitle},
                                                 extraLabel));
        }
      }
    }
  }
  catch (error)
  {
    if (error.message != "invalid") alert("Error saving data.\n\nError: " + error.message + "\nLine: " + error.lineNumber + "\nStack:\n" + error.stack);
  }
  document.getElementById("saveButton").value = "Save";
}

function validateFile(index, data)
{
  var i, j, k, l, inputSources, inputIndex, valid = true, found, magicBytes, combinedData = [], length, titleSpecificData, titleSpecificDataLength, entryDataOffset, baseOffset, entryId, entryOffset, entryLength, entryData, entryIndex, score, flags, element, optionElements, infoElements, headerSize, hashTableOffset, blockOffset, fileTableBlockIndex, fileTableBlockCount, fileTableOffset, tableData, files, fileData, blockCount, startBlock, fileSize, fileData, extraHashBlock, blockHash, fileNameLength;
  
  if (saveFileFormat.combinedInputSource) inputSources = getArray(saveFileFormat.combinedInputSource.inputSource);
  else inputSources = getArray(saveFileFormat.inputSource);
  for (i = 0; i < inputSources.length; i++)
  {
    if (defined(inputSources[i].format)) inputSources[i].format = inputSources[i].format.toLowerCase();
    inputIndex = getIndex(inputSources[i].index, i);
    if (inputIndex == index)
    {
      switch (inputSources[i].format)
      {
        case "xdbf-settings":
          try
          {
            inputSources[i].XDBF = new XDBF(data);
          }
          catch (error)
          {
            alert("Error parsing GPD file.\n\nError: " + error.message + "\nLine: " + error.lineNumber + "\nStack:\n" + error.stack);
            valid = false;
            break;
          }
          if (inputSources[i].titleName && inputSources[i].XDBF.titleName != inputSources[i].titleName)
          {
            alert("GPD file is for the game " + inputSources[i].XDBF.titleName + ", not " + inputSources[i].titleName + ".");
            valid = false;
            break;
          }
          if (!inputSources[i].XDBF.titleSpecificData)
          {
            alert(inputSources[i].name + " does not contain all TitleSpecific data entries.");
            valid = false;
            break;
          }
          data = inputSources[i].XDBF.titleSpecificData;
          element = document.getElementById("gameInfo" + inputIndex);
          if (element)
          {
            infoElements = [];
            clearChildren(element);
            if (inputSources[i].XDBF.titleImageData)
              infoElements.push(addElement("div",
                                           {class: "gameInfoCell"},
                                           addElement("img",
                                                      {class: "pictureImg",
                                                       src: "data:image/png;base64," + encode64(inputSources[i].XDBF.titleImageData),
                                                       title: "Title Image"})));
            infoElements.push(addElement("div",
                                         {class: "gameInfoCell"},
                                         [addElement("div",
                                                     {class: "gameInfoScore",
                                                      title: "Gamerscore"},
                                                     inputSources[i].XDBF.achievementScoreAchieved + " / " + inputSources[i].XDBF.achievementTotalScore),
                                          addElement("div",
                                                     {class: "gameInfoAchievements",
                                                      title: "Achievements"},
                                                      inputSources[i].XDBF.achievementsAchieved + " / " + inputSources[i].XDBF.achievementCount),
                                          addElement("div",
                                                     {class: "gameInfoAchievementIcon",
                                                      title: "Achievements"})]));
            insertElementNodes(element, infoElements);
          }
          break;
        
        case "stfs":
          try
          {
            inputSources[i].STFS = new STFS(data);
            data = null;
          }
          catch (error)
          {
            alert("Error parsing STFS file.\n\nError: " + error.message + "\nLine: " + error.lineNumber + "\nStack:\n" + error.stack);
            valid = false;
            break;
          }
          if (inputSources[i].titleId && toHexStr(inputSources[i].STFS.title_id.toString()) != toHexStr(inputSources[i].titleId, true))
          {
            alert("Save file is not for the game " + inputSources[i].titleName + ". Title ID is " + toHexStr(inputSources[i].STFS.title_id.toString(), false, true) + ", should be " + toHexStr(inputSources[i].titleId, true, true) + ".");
            valid = false;
            break;
          }
          files = getArray(inputSources[i].file);
          for (j = 0; j < files.length; j++)
          {
            if (files[j].required && !inputSources[i].STFS.allfiles[files[j].name])
            {
              alert(files[j].name + " is not present in the save.");
              valid = false;
              break;
            }
          }
          element = document.getElementById("gameInfo" + inputIndex);
          if (element)
          {
            infoElements = [];
            clearChildren(element);
            infoElements.push(addElement("div",
                                         {class: "gameInfoCell"},
                                         addElement("img",
                                                    {class: "pictureImg",
                                                     src: "data:image/png;base64," + encode64(inputSources[i].STFS.titleimage),
                                                     title: "Title Image"})));
            infoElements.push(addElement("div",
                                         {class: "gameInfoCell"},
                                         [addElement("div",
                                                     {class: "gameInfoRow"},
                                                     [addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 "Profile ID:"),
                                                      addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 getHexStr(inputSources[i].STFS.profile_id, 0, 8, false, true))]),
                                          addElement("div",
                                                     {class: "gameInfoRow"},
                                                     [addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 "Device ID:"),
                                                      addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 getHexStr(inputSources[i].STFS.device_id, 0, 20, false, true))]),
                                          addElement("div",
                                                     {class: "gameInfoRow"},
                                                     [addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 "Console ID:"),
                                                      addElement("div",
                                                                 {class: "gameInfoCell"},
                                                                 getHexStr(inputSources[i].STFS.console_id, 0, 5, false, true))])]));
            insertElementNodes(element, infoElements);
          }
          break;
        
        default:
      }
      if (!valid) break;
      if (inputSources[i].format != "stfs")
      {
        magicBytes = getArray(inputSources[i].magicBytes);
        for (j = 0; j < magicBytes.length; j++)
        {
          if (getNum(data, magicBytes[j].position, magicBytes[j].length) != parseInt(magicBytes[j].value))
          {
            alert(inputSources[i].name + " does not appear to be valid.");
            valid = false;
            break;
          }
        }
        if (!valid) break;
        if (inputSources[i].length && data.length != parseInt(inputSources[i].length))
        {
          alert(inputSources[i].name + " does not have the correct length (" + data.length + ", should be " + parseInt(inputSources[i].length) + ").");
          valid = false;
        }
      }
    }
    if (!valid) break;
  }
  if (valid)
  {
    if (inputSources[index].format == "stfs")
    {
      files = getArray(inputSources[index].file);
      /* If only one file, load it now. Otherwise show file selection. */
      if (files.length == 1) saveFileData[index] = loadFile(index, 0);
      else
      {
        optionElements = [];
        element = document.getElementById("fileChoice");
        if (element)
        {
          clearChildren(element);
          for (i = 0; i < files.length; i++)
          {
            optionElements.push(addElement("option",
                                           {value: index + "-" + i},
                                           (files[i].label ? files[i].label : files[i].name)));
          }
          insertElementNodes(element, [addElement("div",
                                                  {class: "fileName"},
                                                  "File Selection"),
                                       addElement("select",
                                                  {id: "fileChoiceSelect"},
                                                  optionElements),
                                       addElement("input",
                                                  {id: "fileSelectButton",
                                                   type: "button",
                                                   value: "Select File",
                                                   onclick: "setSelectFileEvent();"})]);
        }
        clearChildren(document.getElementById("editor"));
        clearChildren(document.getElementById("loadInfoMsg"));
      }
    }
    else saveFileData[index] = data;
    for (i = 0; i < inputSources.length; i++)
    {
      index = getIndex(inputSources[i].index, i);
      if (!defined(saveFileData[index]) || saveFileData[index].length == 0)
      {
        valid = false;
        break;
      }
    }
    /* If all files are present */
    if (valid)
    {
      for (i = 0; i < inputSources.length; i++)
      {
        index = getIndex(inputSources[i].index, i);
        clearChildren(document.getElementById("download" + index));
      }
      if (saveFileFormat.combinedInputSource)
      {
        length = 0;
        for (i = 0; i < inputSources.length; i++)
        {
          index = getIndex(inputSources[i].index, i);
          combinedData[inputSources[i].combinedSource.index] = saveFileData[index].subarray(inputSources[i].combinedSource.offset, parseInt(inputSources[i].combinedSource.offset) + parseInt(inputSources[i].combinedSource.length));
          length += parseInt(inputSources[i].combinedSource.length);
        }
        index = saveFileFormat.combinedInputSource.index;
        saveFileData[index] = new Uint8Array(length);
        length = 0;
        for (i = 0; i < combinedData.length; i++)
        {
          saveFileData[index].set(combinedData[i], length);
          length += combinedData[i].length;
        }
      }
      load();
    }
  }
  else
  {
    saveFileData[index] = [];
    clearChildren(document.getElementById("loadInfoMsg"));
    element = document.getElementById("saveButton");
    if (element) element.style.display = "none";
    for (i = 0; i < inputSources.length; i++)
    {
      clearChildren(document.getElementById("gameInfo" + i));
    }
    clearChildren(document.getElementById("fileChoice"));
    clearChildren(document.getElementById("editor"));
  }
}

function handleFileSelect(selectEvent, index)
{
  var reader = new FileReader(), files = selectEvent.target.files;
  if (files.length == 1)
  {
    reader.onload = function(fileEvent)
                    {
                      setValidateFileEvent(index, new Uint8Array(fileEvent.target.result));
                    };
    reader.readAsArrayBuffer(files[0]);
  }
}

function setSelectGameEvent()
{
  var element = document.getElementById("gameSelectionSelect");
  if (!element || element.value == "") return;
  element = document.getElementById("saveFiles")
  clearChildren(element);
  insertElementNodes(element, addElement("div",
                                         {class: "loading"},
                                         "Loading..."));
  element.scrollIntoView(true);
  clearChildren(document.getElementById("editor"));
  /* Timeout set to ensure the loading message will be rendered before XML loading/parsing happens */
  setTimeout(selectGame, 50);
}

function setSelectFileEvent()
{
  var element = document.getElementById("editor");
  clearChildren(element);
  element = document.getElementById("saveButton");
  if (element) element.style.display = "none";
  element = document.getElementById("loadInfoMsg");
  insertElementNodes(element, addElement("div",
                                         {class: "loading"},
                                         "Reading file..."));
  setTimeout(selectFile, 50);
}

function setValidateFileEvent(index, data)
{
  var i = 0, element = document.getElementById("editor");
  clearChildren(element);
  element = document.getElementById("gameInfo" + i);
  while (element)
  {
    clearChildren(element);
    element = document.getElementById("gameInfo" + ++i);
  }
  element = document.getElementById("fileChoice");
  clearChildren(element);
  element = document.getElementById("saveButton");
  if (element) element.style.display = "none";
  element = document.getElementById("loadInfoMsg");
  clearChildren(element);
  insertElementNodes(element, addElement("div",
                                         {class: "loading"},
                                         "Reading file..."));
  setTimeout(validateFile, 50, index, data);
}

function loadGameXML(url)
{
  var xmlObj, request = new XMLHttpRequest();
  request.open("get", url, false);
  request.send();
  /* Status is 0 for local filesystem requests, ie, XXSE being used offline */
  if (request.status == 0 || (request.status >= 200 && request.status < 300)) xmlObj = parseElement(request.responseXML);
  else xmlObj = null;
  return xmlObj;
}

function parseElement(element)
{
  var i, xmlObj, key, val, count = {}, textonly, ntype;
  
  // COMMENT_NODE
  if (element.nodeType == 7) return null;
  
  // TEXT_NODE CDATA_SECTION_NODE
  if (element.nodeType == 3 || element.nodeType == 4)
  {
    if (element.nodeValue.trim() == "") return null; // ignore white space
    return element.nodeValue;
  }
  
  // parse attributes
  if (element.attributes && element.attributes.length)
  {
    xmlObj = {};
    for (i = 0; i < element.attributes.length; i++)
    {
      key = element.attributes[i].nodeName;
      if (typeof key != "string") continue;
      val = element.attributes[i].value;
      if (!val) continue;
      if (typeof count[key] == "undefined") count[key] = 0;
      count[key]++;
      addNode(xmlObj, key, count[key], val);
    }
  }

  // parse child nodes (recursive)
  if (element.childNodes && element.childNodes.length)
  {
    textonly = true;
    if (xmlObj) textonly = false; // some attributes exists
    for (i = 0; i < element.childNodes.length && textonly; i++)
    {
      ntype = element.childNodes[i].nodeType;
      if (ntype == 3 || ntype == 4) continue;
      textonly = false;
    }
    if (textonly)
    {
      if (!xmlObj) xmlObj = "";
      for (i = 0; i < element.childNodes.length; i++)
      {
        xmlObj += element.childNodes[i].nodeValue;
      }
    }
    else
    {
      if (!xmlObj) xmlObj = {};
      for (i = 0; i < element.childNodes.length; i++)
      {
        key = element.childNodes[i].nodeName;
        if (typeof key != "string") continue;
        val = parseElement(element.childNodes[i]);
        if (!val) continue;
        if (typeof count[key] == "undefined") count[key] = 0;
        count[key]++;
        addNode(xmlObj, key, count[key], val);
      }
    }
  }
  return xmlObj;
}

function addNode(xmlObj, key, count, val)
{
  if (count == 1) xmlObj[key] = val;                     // 1st sibling
  else if (count == 2) xmlObj[key] = [xmlObj[key], val]; // 2nd sibling
  else xmlObj[key][xmlObj[key].length] = val;            // 3rd sibling and more
}

function selectGame()
{
  var i, xmlObj, inputSources, inputElements = [], subElements = [], index, element, attributes;
  saveFileData = [];
  for (i = 0; i < gameSelections.length; i++)
  {
    element = document.getElementById("game-" + gameSelections[i][2])
    if (!element) continue;
    if (element.checked)
    {
      try
      {
        xmlObj = loadGameXML(gameSelections[i][1]);
        if (!xmlObj) throw new Error("network");
        saveFileFormat = xmlObj.saveFileFormat;
      }
      catch (error)
      {
        if (error.message == "network" || error.name == "NETWORK_ERR" || error.number == -0x7FF8FFFB)
        {
          alert("Error reading XML file " + gameSelections[i][1] + ".\n\nNote that Chrome, Opera, and Internet Explorer will not be able to read XML files when XXSE is used offline. Please go to http://absurdlyobfuscated.com/xxse/ or use Firefox.");
        }
        else alert("Error loading game.\n\nError: " + error.message + "\nLine: " + error.lineNumber + "\nStack:\n" + error.stack);
        clearChildren(document.getElementById("saveFiles"));
        return;
      }
      break;
    }
  }
  if (!saveFileFormat)
  {
    clearChildren(document.getElementById("saveFiles"));
    return;
  }
  if (saveFileFormat.combinedInputSource) inputSources = getArray(saveFileFormat.combinedInputSource.inputSource);
  else inputSources = getArray(saveFileFormat.inputSource);
  subElements.push(addElement("div",
                              {class: "groupTitle"},
                              addElement("div",
                                         {class: "groupTitleText"},
                                         gameSelections[i][0] + " Save")));
  for (i = 0; i < inputSources.length; i++)
  {
    attributes = {class: "fileItem"};
    if (inputSources[i].info) attributes.title = inputSources[i].info;
    index = getIndex(inputSources[i].index, i);
    inputElements.push(addElement("div",
                                  attributes,
                                  [addElement("div",
                                              {class: "fileName"},
                                              inputSources[i].name),
                                   addElement("input",
                                              {class: "file",
                                               type: "file",
                                               id: "file" + index,
                                               onchange: "handleFileSelect(event, " + index + ");"}),
                                   addElement("div",
                                              {class: "fileDownload",
                                               id: "download" + index})]));
    inputElements.push(addElement("div",
                                  {class: "gameInfo",
                                   id: "gameInfo" + index}));
  }
  inputElements.push(addElement("div",
                                {id: "fileChoice"}));
  inputElements.push(addElement("input",
                                {id: "saveButton",
                                 type: "button",
                                 onclick: "setSaveEvent();",
                                 style: "display: none;",
                                 value: "Save"}));
  inputElements.push(addElement("div",
                                {id: "loadInfoMsg"}));
  subElements.push(addElement("div",
                              {class: "saveFilesContent"},
                              inputElements));
  element = document.getElementById("saveFiles");
  clearChildren(element);
  insertElementNodes(element, addElement("div",
                                         {class: "group"},
                                         subElements));
  element.scrollIntoView(true);
}

function selectFile()
{
  var element, indexes, inputIndex, fileIndex;
  element = document.getElementById("fileChoiceSelect");
  if (!element) return;
  indexes = element.value.split("-");
  if (indexes.length != 2) return;
  inputIndex = indexes[0];
  fileIndex = indexes[1];
  saveFileData[inputIndex] = loadFile(inputIndex, fileIndex);
  load();
}

function loadFile(inputIndex, fileIndex)
{
  var i, inputSources, files, data, magicBytes;
  if (saveFileFormat.combinedInputSource) inputSources = getArray(saveFileFormat.combinedInputSource.inputSource);
  else inputSources = getArray(saveFileFormat.inputSource);
  inputSources[inputIndex].fileIndex = fileIndex;
  files = getArray(inputSources[inputIndex].file)
  data = inputSources[inputIndex].STFS.read_file(inputSources[inputIndex].STFS.allfiles[files[fileIndex].name], -1);
  magicBytes = getArray(files[fileIndex].magicBytes);
  for (i = 0; i < magicBytes.length; i++)
  {
    if (getNum(data, magicBytes[i].position, magicBytes[i].length) != parseInt(magicBytes[i].value))
    {
      alert("File " + files[fileIndex].name + " in " + inputSources[inputIndex].name + " does not appear to be valid.");
      return null;
    }
  }
  if (files[fileIndex].length && data.length != parseInt(files[fileIndex].length))
  {
    alert(inputSources[inputIndex].name + " does not have the correct length (" + data.length + ", should be " + parseInt(files[fileIndex].length) + ").");
    return null;
  }
  return data;
}

function selectRow(row)
{
  var i, element;
  for (i = 0; i < gameSelections.length; i++)
  {
    element = document.getElementById("game-" + gameSelections[i][2] + "-label")
    if (!element) continue;
    if (i == row)
    {
      element.className = "gameSelectionLabelSelected game-" + gameSelections[i][2];
      element = document.getElementById("selectGameButton");
      if (element) element.focus();
    }
    else element.className = "gameSelectionLabel game-" + gameSelections[i][2];
  }
}

function init()
{
  var i, optionElements = [], element;
  if (typeof FileReader == "undefined")
  {
    document.getElementById("gameSelection").innerHTML = "Your browser does not support <a href=\"https://developer.mozilla.org/en/DOM/FileReader\">FileReader</a>. Try using <a href=\"http://getfirefox.com/\">Firefox</a> or <a href=\"https://www.google.com/chrome\">Chrome</a>.";
    return;
  }
  element = document.getElementById("gameSelection");
  clearChildren(element);
  for (i = 0; i < gameSelections.length; i++)
  {
    optionElements.push(addElement("div",
                                   {class: "gameSelectionRow"},
                                   [addElement("input",
                                               {id: "game-" + gameSelections[i][2],
                                                class: "gameSelectionInput",
                                                type: "radio",
                                                name: "game"}),
                                    addElement("label",
                                               {id: "game-" + gameSelections[i][2] + "-label",
                                                class: "gameSelectionLabel game-" + gameSelections[i][2],
                                                for: "game-" + gameSelections[i][2],
                                                onclick: "selectRow(" + i + ");"},
                                               gameSelections[i][0])]));
  }
  insertElementNodes(element, addElement("div",
                                         {class: "group"},
                                         [addElement("div",
                                                     {class: "groupTitle"},
                                                     addElement("div",
                                                                {class: "groupTitleText"},
                                                                "Game Selection")),
                                          addElement("div",
                                                     {class: "gameSelectionContent"},
                                                     [addElement("div",
                                                                 {id: "gameSelectionSelect"},
                                                                 optionElements),
                                                      addElement("input",
                                                                 {id: "selectGameButton",
                                                                  type: "button",
                                                                  value: "Select Game",
                                                                  onclick: "setSelectGameEvent();"})]),
                                          addElement("div",
                                                     {id: "saveFiles"})]));
}
